
/*
 * max3353_mcf532x - Maxim MAX3353 USB OTG Churge Pump with
 *                   Switchable Pullup/Pulldown Registers,
 *                   talking with MCF532x OTG module.
 *
 *      Andrey Butok
 *      Copyright Freescale Semiconductor, Inc. 2006
 *
 *      General structure is based on:
 *       isp1301_omap.c - David Brownell
 *      OTG statemchine is based on MPC code from:
 *       Leo Li<LeoLi@freescale.com>
 *       Jerry Huang<Chang-Ming.Huang@freescale.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ***************************************************************************
 * Changes:
 *   v0.002	29 September 2006	Andrey Butok
 *   		OTG support added.
 *   v0.001	12 July 2006		Andrey Butok
 *   		Development version  - developed on 2.6.17.1 kernel.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/smp_lock.h>
#include <linux/proc_fs.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/reboot.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/usb.h>
#include <linux/device.h>
#include <linux/usb_ch9.h>
#include <linux/usb_gadget.h>
#include <linux/workqueue.h>
#include <linux/time.h>
#include <linux/platform_device.h>
#include <linux/i2c.h>

#include <asm/m532xsim.h>

#include "max3353_mcf532x.h"

#if 0
#define DEBUG 		1
#define VERBOSE 	1
#endif

#ifdef DEBUG
#define DBG(fmt, args...) 	printk( "[%s]  " fmt , \
				__FUNCTION__, ## args)
#else
#define DBG(fmt, args...)	do{}while(0)
#endif

#ifdef VERBOSE
#define VDBG		DBG
#else
#define VDBG(stuff...)	do{}while(0)
#endif

#define ERR(format, arg...) \
printk(KERN_ERR "%s:%s: " format "\n" , __FILE__,  __FUNCTION__ , ## arg)
#define WARN(stuff...)		printk(KERN_WARNING "max3353: " stuff)
#define INFO(stuff...)		printk(KERN_INFO "max3353: " stuff)

#define CONFIG_USB_OTG_DEBUG_FILES
#define DRIVER_VERSION "Revision: 0.002"
#define DRIVER_AUTHOR "Andrey Butok"
#define DRIVER_DESC "Freescale MAX3353 OTG Driver"
#define DRIVER_INFO DRIVER_VERSION " " DRIVER_DESC

MODULE_DESCRIPTION
    ("MAX3353 USB OTG Churge Pump with Switchable Pullup/Pulldown Registers Driver");

static const char otg_dr_name[] = "max3353_mcf532x";
#define	DRIVER_NAME	(max3353_driver.driver.name)

static struct max3353 *the_transceiver = NULL;
static int srp_wait_done;

/* Driver specific timers */
static struct max3353_otg_timer *b_data_pulse_tmr, *b_vbus_pulse_tmr,
    *b_srp_wait_tmr, *a_wait_enum_tmr;

static struct list_head active_timers;

/*--------------------I2C staff-----------------------------------------------------*/

/* only two addresses possible */
static unsigned short normal_i2c[] = { 0x2C, 0x2D, I2C_CLIENT_END };

I2C_CLIENT_INSMOD;

static struct i2c_driver max3353_driver;

static inline u8
max3353_read_reg(struct max3353 *max, u8 reg)
{
	return i2c_smbus_read_byte_data(&max->client, reg);
}

static inline u8
max3353_write_reg(struct max3353 *max, u8 reg, u8 data)
{
	return i2c_smbus_write_byte_data(&max->client, reg, data);
}

/* prototype declaration */
static void max3353_otg_add_timer(void *timer);
static void max3353_otg_del_timer(void *timer);

/* -------------------------------------------------------------*/
/* Operations that will be called from OTG Finite State Machine */

/* Charge vbus for vbus pulsing in SRP */
static void
max3353_otg_chrg_vbus(int on)
{
	u8 tmp = max3353_read_reg(the_transceiver, MAX3353_CONTROL_2);

	VDBG(" %d\n", on);
	if (on) {
		max3353_write_reg(the_transceiver,
				  MAX3353_CONTROL_2,
				  tmp | MAX3353_CONTROL_2_VBUS_CHG2);
	} else {
		max3353_write_reg(the_transceiver,
				  MAX3353_CONTROL_2,
				  tmp & ~MAX3353_CONTROL_2_VBUS_CHG2);
	}
}

/* Discharge vbus through a resistor to ground */
static void
max3353_otg_dischrg_vbus(int on)
{
	u8 tmp = max3353_read_reg(the_transceiver, MAX3353_CONTROL_2);

	VDBG(" %d\n", on);
	if (on) {
		max3353_write_reg(the_transceiver,
				  MAX3353_CONTROL_2,
				  tmp | MAX3353_CONTROL_2_VBUS_DISCHG);

	} else {
		max3353_write_reg(the_transceiver,
				  MAX3353_CONTROL_2,
				  tmp & ~MAX3353_CONTROL_2_VBUS_DISCHG);
	}
}

/* A-device driver vbus, controlled through PP bit in PORTSC */
static void
max3353_otg_drv_vbus(int on)
{
	u8 tmp = max3353_read_reg(the_transceiver, MAX3353_CONTROL_2);
	VDBG(" %d\n", on);
	if (on) {
		MCF_USB0_PORTSC =
		    (MCF_USB0_PORTSC & ~PORTSC_W1C_BITS) | PORTSC_PORT_POWER;
		max3353_write_reg(the_transceiver, MAX3353_CONTROL_2,
				  tmp | MAX3353_CONTROL_2_VBUS_DRV);

	} else {
		MCF_USB0_PORTSC =
		    MCF_USB0_PORTSC & ~PORTSC_W1C_BITS & ~PORTSC_PORT_POWER;
		max3353_write_reg(the_transceiver, MAX3353_CONTROL_2,
				  tmp & ~MAX3353_CONTROL_2_VBUS_DRV);
	}
}

/* Pull-up D+, signalling connect by periperal. Also used in
 * data-line pulsing in SRP */
static void
max3353_otg_loc_conn(int on)
{
	u8 tmp = max3353_read_reg(the_transceiver, MAX3353_CONTROL_1);

	VDBG(" %d\n", on);
	if (on) {
		max3353_write_reg(the_transceiver,
				  MAX3353_CONTROL_1,
				  tmp | MAX3353_CONTROL_1_DP_PULLUP);
	} else {
		max3353_write_reg(the_transceiver,
				  MAX3353_CONTROL_1,
				  tmp & ~MAX3353_CONTROL_1_DP_PULLUP);
	}
}

/* Generate SOF by host.  This is controlled through suspend/resume the
 * port.  In host mode, controller will automatically send SOF.
 * Suspend will block the data on the port. */
static void
max3353_otg_loc_sof(int on)
{
	u32 tmpval;
	VDBG(" %d\n", on);
	tmpval = MCF_USB0_PORTSC & ~PORTSC_W1C_BITS;
	if (on) {
		tmpval |= PORTSC_PORT_FORCE_RESUME;
	} else {
		tmpval |= PORTSC_PORT_SUSPEND;
	}
	MCF_USB0_PORTSC = tmpval;
}

/* Start SRP pulsing by data-line pulsing, followed with v-bus pulsing. */
static void
max3353_otg_start_pulse(void)
{
	VDBG("\n");
	srp_wait_done = 0;
	max3353_otg_loc_conn(1);
	max3353_otg_add_timer(b_data_pulse_tmr);
}

static void max3353_otg_pulse_vbus(void);

static void
b_data_pulse_end(unsigned long foo)
{
	VDBG("\n");
	max3353_otg_loc_conn(0);
	/* Do VBUS pulse after data pulse */
	max3353_otg_pulse_vbus();
}

static void
max3353_otg_pulse_vbus(void)
{
	VDBG("\n");
	srp_wait_done = 0;
	max3353_otg_chrg_vbus(1);
	/* start the timer to end vbus charge */
	max3353_otg_add_timer(b_vbus_pulse_tmr);
}

static void
b_vbus_pulse_end(unsigned long foo)
{
	VDBG("\n");
	max3353_otg_chrg_vbus(0);

	/* As USB3300 using the same a_sess_vld and b_sess_vld voltage
	 * we need to discharge the bus for a while to distinguish
	 * residual voltage of vbus pulsing and A device pull up */
	max3353_otg_dischrg_vbus(1);
	max3353_otg_add_timer(b_srp_wait_tmr);
}

static void
b_srp_end(unsigned long foo)
{
	VDBG("\n");
	max3353_otg_dischrg_vbus(0);
	srp_wait_done = 1;

	if ((the_transceiver->otg.state == OTG_STATE_B_SRP_INIT) &&
	    the_transceiver->fsm.b_sess_vld)
		the_transceiver->fsm.b_srp_done = 1;
}

/* Workaround for a_host suspending too fast.  When a_bus_req=0,
 * a_host will start by SRP.  It needs to set b_hnp_enable before
 * actually suspending to start HNP */
static void
a_wait_enum(unsigned long foo)
{
	VDBG("\n");
	if (!the_transceiver->otg.host->b_hnp_enable)
		max3353_otg_add_timer(a_wait_enum_tmr);
	else
		otg_fsm_statemachine(&the_transceiver->fsm);
}

/* ------------------------------------------------------*/

/* The timeout callback function to set time out bit */
static void
set_tmout(unsigned long indicator)
{
	*(int *) indicator = 1;
}

/* Initialize timers */
static void
max3353_otg_init_timers(struct otg_fsm *fsm)
{
	/* FSM used timers */
	fsm->a_wait_vrise_tmr =
	    max3353_otg_timer_initializer(&set_tmout, TA_WAIT_VRISE,
					  (unsigned long) &fsm->
					  a_wait_vrise_tmout);
	fsm->a_wait_bcon_tmr =
	    max3353_otg_timer_initializer(&set_tmout, TA_WAIT_BCON,
					  (unsigned long) &fsm->
					  a_wait_bcon_tmout);
	fsm->a_aidl_bdis_tmr =
	    max3353_otg_timer_initializer(&set_tmout, TA_AIDL_BDIS,
					  (unsigned long) &fsm->
					  a_aidl_bdis_tmout);
	fsm->b_ase0_brst_tmr =
	    max3353_otg_timer_initializer(&set_tmout, TB_ASE0_BRST,
					  (unsigned long) &fsm->
					  b_ase0_brst_tmout);
	fsm->b_se0_srp_tmr =
	    max3353_otg_timer_initializer(&set_tmout, TB_SE0_SRP,
					  (unsigned long) &fsm->b_se0_srp);
	fsm->b_srp_fail_tmr =
	    max3353_otg_timer_initializer(&set_tmout, TB_SRP_FAIL,
					  (unsigned long) &fsm->b_srp_done);
	fsm->a_wait_enum_tmr =
	    max3353_otg_timer_initializer(&a_wait_enum, 10,
					  (unsigned long) &fsm);

	/* device driver used timers */
	b_srp_wait_tmr =
	    max3353_otg_timer_initializer(&b_srp_end, TB_SRP_WAIT, 0);
	b_data_pulse_tmr =
	    max3353_otg_timer_initializer(&b_data_pulse_end, TB_DATA_PLS, 0);
	b_vbus_pulse_tmr =
	    max3353_otg_timer_initializer(&b_vbus_pulse_end, TB_VBUS_PLS, 0);

}

/* Add timer to timer list */
static void
max3353_otg_add_timer(void *gtimer)
{
	struct max3353_otg_timer *timer = (struct max3353_otg_timer *) gtimer;
	struct max3353_otg_timer *tmp_timer;

	/* Check if the timer is already in the active list, 
	 * if so update timer count */
	list_for_each_entry(tmp_timer, &active_timers, list)
	    if (tmp_timer == timer) {
		timer->count = timer->expires;
		return;
	}
	timer->count = timer->expires;
	list_add_tail(&timer->list, &active_timers);
}

/* Remove timer from the timer list; clear timeout status */
static void
max3353_otg_del_timer(void *gtimer)
{
	struct max3353_otg_timer *timer = (struct max3353_otg_timer *) gtimer;
	struct max3353_otg_timer *tmp_timer, *del_tmp;

	list_for_each_entry_safe(tmp_timer, del_tmp, &active_timers, list)
	    if (tmp_timer == timer)
		list_del(&timer->list);
}

/* Reduce timer count by 1, and find timeout conditions.
 * Called by max3353_irq_otg 1ms timer interrupt
 */
static int
max3353_otg_tick_timer(void)
{
	struct max3353_otg_timer *tmp_timer, *del_tmp;
	int expired = 0;

	list_for_each_entry_safe(tmp_timer, del_tmp, &active_timers, list) {
		tmp_timer->count--;
		/* check if timer expires */
		if (!tmp_timer->count) {
			list_del(&tmp_timer->list);
			tmp_timer->function(tmp_timer->data);
			expired = 1;
		}
	}

	return expired;
}

/* Reset controller, not reset the bus */
static void
otg_reset_controller(void)
{
	u32 command;
	int timeout;

	VDBG("\n");

	command = MCF_USB0_USBCMD;
	command |= MCF_USB_USBCMD_RST;
	MCF_USB0_USBCMD = command;

	/* Wait reset completed */
	timeout = 500;
	while (timeout-- && (MCF_USB0_USBCMD & 0x2))
		udelay(1);
	if (timeout <= 0)
		ERR("%s - USBCMD_RST never clear. Timeout is %d \n",
		    __FUNCTION__, timeout);
}

/* Call suspend/resume routines in host driver */
static int
max3353_otg_start_host(struct otg_fsm *fsm, int on)
{
	struct otg_transceiver *xceiv = fsm->transceiver;
	struct device *dev = xceiv->host->controller;
	struct max3353 *max = container_of(xceiv, struct max3353, otg);
	u32 retval = 0;

#ifdef DEBUG
	if (max->host_working) {
		VDBG("host working!\n");
	} else {
		VDBG("host is not working!\n");
	}
#endif

	if (!xceiv->host) {
		return -ENODEV;
	}

	if (on) {
		/*start usb host controller */
		if (max->host_working)
			goto end;
		else {
			otg_reset_controller();
			max3353_write_reg(max, MAX3353_CONTROL_1,
					  MAX3353_CONTROL_1_DP_PULLDWN |
					  MAX3353_CONTROL_1_DM_PULLDWN);
			VDBG("host on......\n");
			if (dev->driver->resume) {
				retval = dev->driver->resume(dev /*, 0 */ );
				if (fsm->id) {
					/* default-b */
					max3353_otg_drv_vbus(1);
				}
			}
			max->host_working = 1;
		}
	} else {
		/*stop usb host controller */
		if (!max->host_working)
			goto end;
		else {
			VDBG("host off......\n");
			if (dev && dev->driver) {
				retval =
				    dev->driver->suspend(dev,
							 PMSG_SUSPEND /*, 0 */
							 );
				if (fsm->id)
					/* default-b */
					max3353_otg_drv_vbus(0);
			}
			max->host_working = 0;
		}
	}
      end:
	return retval;
}

/* Call suspend and resume function in udc driver
 * to stop and start udc driver.
 */
static int
max3353_otg_start_gadget(struct otg_fsm *fsm, int on)
{
	struct otg_transceiver *xceiv = fsm->transceiver;
	struct device *dev = &xceiv->gadget->dev;

	if (!xceiv->gadget)
		return -ENODEV;

	VDBG("starting gadget %s \n", on ? "on" : "off");

	if (on) {

		fsm->b_bus_suspend = 0;
		fsm->a_bus_suspend = 0;
		dev->driver->resume(dev /*, 0 */ );
	} else {
		dev->driver->suspend(dev, PMSG_SUSPEND /* 0, 0 */ );
	}

	return 0;
}

/* Set OTG port power,only for B-device */
static int
max3353_otg_set_power(struct otg_transceiver *otg_p, unsigned mA)
{
	if (!the_transceiver)
		return -ENODEV;
	if (otg_p->state == OTG_STATE_B_PERIPHERAL)
		INFO("max3353 OTG:Draw %d mA\n", mA);

	return 0;
}

/* B-device start SRP */
static int
max3353_otg_start_srp(struct otg_transceiver *otg_p)
{
	struct max3353 *otg_dev = container_of(otg_p, struct max3353, otg);

	if (!otg_p || otg_dev != the_transceiver
	    || otg_p->state != OTG_STATE_B_IDLE)
		return -ENODEV;

	VDBG("\n");
	otg_dev->fsm.b_bus_req = 1;
	otg_fsm_statemachine(&otg_dev->fsm);

	return 0;
}

/* A_host suspend will call this function to start hnp */
static int
max3353_otg_start_hnp(struct otg_transceiver *otg_p)
{
	struct max3353 *otg_dev = container_of(otg_p, struct max3353, otg);

	if (!otg_p || otg_dev != the_transceiver)
		return -ENODEV;

	VDBG("\n");
	/* clear a_bus_req to enter a_suspend state */
	otg_dev->fsm.a_bus_req = 0;
	otg_fsm_statemachine(&otg_dev->fsm);

	return 0;
}

/* Interrupt handler.  OTG/host/peripheral share the same int line.
 * OTG driver clears OTGSC interrupts and leaves USB interrupts
 * intact.  It needs to have knowledge of some USB interrupts
 * such as port change. */

irqreturn_t
max3353_irq(int irq, void *dev_id, struct pt_regs * regs)
{
	struct max3353 *max = (struct max3353 *) dev_id;
	struct otg_fsm *fsm = &((struct max3353 *) dev_id)->fsm;
	int trigger = 0;
	u8 max_int_src;
	u8 max_stat = 0;

	/* Clear interrupt flag. */
	MCF_EPORT_EPFR = MCF_EPORT_EPFR_EPF6;

	/* Save Interrupt Latch register */
	max_int_src = max3353_read_reg(max, MAX3353_INTERRUPT_LATCH);

	/* process OTG interrupts */
	while (max3353_read_reg(max, MAX3353_STATUS) != max_stat) {
		max_stat = max3353_read_reg(max, MAX3353_STATUS);

		VDBG("\n !!!Interrupt!!! MAX3353_INTERRUPT_LATCH=0x%x MAX3353_STATUS=0x%x MAX3353_INTERRUPT_MASK=0x%x MAX3353_INTERRUPT_EDGE=0x%x\n\n", max_int_src, max_stat, max3353_read_reg(max, MAX3353_INTERRUPT_MASK), max3353_read_reg(max, MAX3353_INTERRUPT_EDGE));

#ifdef	CONFIG_USB_OTG

		max->delay_id =
		    (max_stat & MAX3353_STATUS_SESSION_ID_FLOAT) ? 1 : 0;
		if (max->delay_id != fsm->id)
			max->delay_id_cnt = 400;
#endif

		fsm->a_sess_vld =
		    (max_stat & MAX3353_STATUS_SESSION_VALID) ? 1 : 0;
		if ((fsm->transceiver->state == OTG_STATE_A_IDLE)
		    && fsm->a_sess_vld)
			fsm->a_srp_det = 1;

		fsm->b_sess_vld =
		    (max_stat & MAX3353_STATUS_SESSION_VALID) ? 1 : 0;

		if (fsm->b_sess_vld)
			max3353_write_reg(max, MAX3353_INTERRUPT_EDGE,
					  max3353_read_reg(max,
							   MAX3353_INTERRUPT_EDGE)
					  &
					  ~MAX3353_INTERRUPT_EDGE_SESSION_VALID_ED);
		else
			max3353_write_reg(max, MAX3353_INTERRUPT_EDGE,
					  max3353_read_reg(max,
							   MAX3353_INTERRUPT_EDGE)
					  |
					  MAX3353_INTERRUPT_EDGE_SESSION_VALID_ED);

		fsm->a_vbus_vld =
		    (max_stat & MAX3353_STATUS_VBUS_VALID) ? 1 : 0;

#if 1
		if ((fsm->a_vbus_vld == 0) && (fsm->id == 0)) {
			if (fsm->transceiver->state != OTG_STATE_A_WAIT_VFALL)
				/* Warning insted of transfer to Error state. */
				dev_warn(&max->client.dev,
					 "The USB Device is sinking too much current!!\n");
			fsm->a_vbus_vld = 1;
		}
#endif

		/* SRP done */
		if ((the_transceiver->otg.state == OTG_STATE_B_SRP_INIT) &&
		    fsm->b_sess_vld && srp_wait_done)
			fsm->b_srp_done = 1;

		fsm->b_sess_end =
		    (max_stat & MAX3353_STATUS_SESSION_END) ? 1 : 0;

		if (max_int_src & MAX3353_INTERRUPT_LATCH_A_HNP_RQ
		    /*or DP_SRP */ ) {
			fsm->a_srp_det = 1;
			VDBG("Data pulse int \n");
		}
		trigger = 1;
	}

	/* Invoke statemachine until state is stable */
	while (trigger) {
		trigger = otg_fsm_statemachine(fsm);
	}

	return IRQ_HANDLED;
}

#define MAX3353_DELAY_A_CONN_MS		(400)
#define MAX3353_DELAY_B_BUS_SUSPEND_MS	(800)

irqreturn_t
max3353_irq_otg(int irq, void *dev_id, struct pt_regs * regs)
{
	struct max3353 *max = (struct max3353 *) dev_id;
	struct otg_fsm *fsm = &max->fsm;
	struct otg_transceiver *otg = &((struct max3353 *) dev_id)->otg;
	u32 otg_int_src, usb_int_src, otg_sc;
	int trigger = 0;
	int tmp;

	usb_int_src = MCF_USB0_USBSTS;
	otg_sc = MCF_USB0_OTGSC;
	otg_int_src = otg_sc & OTGSC_INTSTS_MASK & (otg_sc >> 8);

	/* Only clear otg interrupts */
	MCF_USB0_OTGSC |= otg_sc & OTGSC_INTSTS_MASK;

	/* Process OTG interrupts */
	if (otg_int_src) {
		if (otg_int_src & OTGSC_INTSTS_1MS) {
			trigger = max3353_otg_tick_timer();

#ifdef	CONFIG_USB_OTG
			if (max->delay_id_cnt) {
				max->delay_id_cnt--;
				if (max->delay_id_cnt == 0) {
					trigger = 1;

					fsm->id = max->delay_id;
					otg->default_a = (fsm->id == 0);
					if (otg->host)
						otg->host->is_b_host = fsm->id;
					if (otg->gadget)
						otg->gadget->is_a_peripheral =
						    !fsm->id;
				}
			}
#endif
			if (max->delay_a_conn_cnt) {
				max->delay_a_conn_cnt--;
				if (max->delay_a_conn_cnt == 0) {
					trigger = 1;
					fsm->a_conn = max->delay_a_conn;
				}
			}

			if (max->delay_b_bus_suspend_cnt) {
				max->delay_b_bus_suspend_cnt--;
				if (max->delay_b_bus_suspend_cnt == 0) {
					trigger = 1;
					fsm->b_bus_suspend =
					    max->delay_b_bus_suspend;
				}
			}

		}
	}

	/* process USB interrupts */
	if ((usb_int_src & USB_STS_PORT_CHANGE)
	    && (fsm->protocol == OTG_FSM_PROTO_HOST)) {
		/* Device resume do not generate statemachine change */
		if (MCF_USB0_PORTSC & PORTSC_PORT_FORCE_RESUME) {
			if (otg->default_a) {
				fsm->b_bus_resume = 1;
				trigger = 1;
			} else {
				fsm->a_bus_resume = 1;
				trigger = 1;
			}
		}

		tmp = (MCF_USB0_PORTSC & PORTSC_CURRENT_CONNECT_STATUS) ? 1 : 0;
		if (otg->default_a && (fsm->b_conn != tmp)) {
			VDBG("Coonection b_conn=%d \n", tmp);
			fsm->b_conn = tmp;
			trigger = 1;
		} else if (!otg->default_a) {
			max->delay_a_conn = tmp;
			if (fsm->a_conn != tmp) {
				VDBG("Coonection a_conn=%d \n", tmp);
				if (tmp) {
					fsm->a_conn = tmp;
					max->delay_a_conn_cnt = 0;
				} else {
					max->delay_a_conn = tmp;
					max->delay_a_conn_cnt =
					    MAX3353_DELAY_A_CONN_MS;
				}
				trigger = 1;
			}
		}
	}

	/* Workaround: sometimes CSC bit will lost.  We change to
	 * polling CCS bit for connect change */
	if (fsm->protocol == OTG_FSM_PROTO_GADGET) {
		if (usb_int_src & USB_STS_DCSUSPEND) {
			VDBG("peripheral detected suspend \n");
			if (otg->default_a) {
				/* A-device detects B suspend */
				max->delay_b_bus_suspend = 1;
				max->delay_b_bus_suspend_cnt
				    = MAX3353_DELAY_B_BUS_SUSPEND_MS;
			} else
				/* B-device detects A suspend */
				fsm->a_bus_suspend = 1;
			trigger = 1;
		} else if (usb_int_src & USB_STS_PORT_CHANGE) {
			VDBG("peripheral resumed \n");
			if (otg->default_a) {
				fsm->b_bus_suspend = 0;
				max->delay_b_bus_suspend_cnt = 0;
			} else
				fsm->a_bus_suspend = 0;
			trigger = 1;
		}
	}

	/* Invoke statemachine until state is stable */
	while (trigger) {
		trigger = otg_fsm_statemachine(fsm);
	}

	return IRQ_HANDLED;
}

static struct otg_fsm_ops max3353_otg_ops = {
	.chrg_vbus = max3353_otg_chrg_vbus,
	.drv_vbus = max3353_otg_drv_vbus,
	.loc_conn = max3353_otg_loc_conn,
	.loc_sof = max3353_otg_loc_sof,
	.start_pulse = max3353_otg_start_pulse,

	.add_timer = max3353_otg_add_timer,
	.del_timer = max3353_otg_del_timer,

	.start_host = max3353_otg_start_host,
	.start_gadget = max3353_otg_start_gadget,
};

/*-------------------------------------------------------------------------
		PROC File System Support
-------------------------------------------------------------------------*/
#ifdef CONFIG_USB_OTG_DEBUG_FILES

#include <linux/seq_file.h>

static const char proc_filename[] = "driver/max3353_mcf532x";

static int
otg_proc_read(char *page, char **start, off_t off, int count,
	      int *eof, void *_dev)
{
	struct otg_fsm *fsm = &the_transceiver->fsm;
	char *buf = page;
	char *next = buf;
	unsigned size = count;
	unsigned long flags;
	int t;
	u32 tmp_reg;

	if (off != 0)
		return 0;

	spin_lock_irqsave(&fsm->lock, flags);

	/* ------basic driver infomation ---- */
	t = scnprintf(next, size,
		      DRIVER_DESC "\n" "max3353_mcf532x version: %s\n\n",
		      DRIVER_VERSION);
	size -= t;
	next += t;

	/* ------ Registers ----- */
	tmp_reg = MCF_USB0_OTGSC;
	t = scnprintf(next, size, "OTGSC reg: %x\n", tmp_reg);
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_PORTSC;
	t = scnprintf(next, size, "PORTSC reg: %x\n", tmp_reg);
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_USBMODE;
	t = scnprintf(next, size, "USBMODE reg: %x\n", tmp_reg);
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_USBCMD;
	t = scnprintf(next, size, "USBCMD reg: %x\n", tmp_reg);
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_USBSTS;
	t = scnprintf(next, size, "USBSTS reg: %x\n", tmp_reg);
	size -= t;
	next += t;

	/* ------ State ----- */
	t = scnprintf(next, size,
		      "OTG state: %s\n\n",
		      otg_fsm_state_string(the_transceiver->otg.state));
	size -= t;
	next += t;

	/* ------ State Machine Variables ----- */
	t = scnprintf(next, size, "a_bus_req: %d\n", fsm->a_bus_req);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_bus_req: %d\n", fsm->b_bus_req);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_bus_resume: %d\n", fsm->a_bus_resume);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_bus_suspend: %d\n", fsm->a_bus_suspend);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_conn: %d\n", fsm->a_conn);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_sess_vld: %d\n", fsm->a_sess_vld);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_srp_det: %d\n", fsm->a_srp_det);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_vbus_vld: %d\n", fsm->a_vbus_vld);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_bus_resume: %d\n", fsm->b_bus_resume);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_bus_suspend: %d\n", fsm->b_bus_suspend);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_conn: %d\n", fsm->b_conn);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_se0_srp: %d\n", fsm->b_se0_srp);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_sess_end: %d\n", fsm->b_sess_end);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_sess_vld: %d\n", fsm->b_sess_vld);
	size -= t;
	next += t;

	t = scnprintf(next, size, "id: %d\n", fsm->id);
	size -= t;
	next += t;

	spin_unlock_irqrestore(&fsm->lock, flags);

	*eof = 1;
	return count - size;
}

#define create_proc_file()	create_proc_read_entry(proc_filename, \
				0, NULL, otg_proc_read, NULL)

#define remove_proc_file()	remove_proc_entry(proc_filename, NULL)

#else				/* !CONFIG_USB_OTG_DEBUG_FILES */

#define create_proc_file()	do {} while (0)
#define remove_proc_file()	do {} while (0)

#endif				/*CONFIG_USB_OTG_DEBUG_FILES */

/*-------------------------------------------------------------------------*/

static inline const char *
state_name(struct max3353 *max)
{
	return otg_fsm_state_string(max->otg.state);
}

#ifdef	DEBUG
static void
dump_regs(struct max3353 *max, const char *label)
{
	u8 man0 = max3353_read_reg(max, MAX3353_MANUFACTURER_0);
	u8 man1 = max3353_read_reg(max, MAX3353_MANUFACTURER_1);
	u8 man2 = max3353_read_reg(max, MAX3353_MANUFACTURER_2);
	u8 man3 = max3353_read_reg(max, MAX3353_MANUFACTURER_3);
	u8 prod0 = max3353_read_reg(max, MAX3353_PRODUCT_ID_0);
	u8 prod1 = max3353_read_reg(max, MAX3353_PRODUCT_ID_1);
	u8 prod2 = max3353_read_reg(max, MAX3353_PRODUCT_ID_2);
	u8 prod3 = max3353_read_reg(max, MAX3353_PRODUCT_ID_3);
	u8 control1 = max3353_read_reg(max, MAX3353_CONTROL_1);
	u8 control2 = max3353_read_reg(max, MAX3353_CONTROL_2);
	u8 status = max3353_read_reg(max, MAX3353_STATUS);
	u8 int_mask = max3353_read_reg(max, MAX3353_INTERRUPT_MASK);
	u8 int_edge = max3353_read_reg(max, MAX3353_INTERRUPT_EDGE);

	pr_info("max3353 dump: %s %s\n", label, state_name(max));
	pr_info("\t\t Man| 0x%02X 0x%02X 0x%02X 0x%02X\n", man0, man1, man2,
		man3);
	pr_info("\t\t Pro| 0x%02X 0x%02X 0x%02X 0x%02X\n", prod0, prod1, prod2,
		prod3);
	pr_info("\t\t Con| 0x%02X 0x%02X 0x%02X\n", control1, control2, status);
	pr_info("\t\t iMEL 0x%02X 0x%02X \n", int_mask, int_edge);
}
#endif

/*-------------------------------------------------------------------------*/

static struct max3353 *the_transceiver;

static void
max3353_release(struct device *dev)
{
	struct max3353 *max;

	max = container_of(dev, struct max3353, client.dev);

	if (max->i2c_release)
		max->i2c_release(dev);
	kfree(max);
}

static void
max3353_power_down(struct max3353 *max)
{
	max->otg.state = OTG_STATE_UNDEFINED;

	/* Set shutdown mode. */
	max3353_write_reg(max, MAX3353_CONTROL_2,
			  max3353_read_reg(max, MAX3353_CONTROL_2) | 0x1);
}

static void
max3353_power_up(struct max3353 *max)
{
	/* Set operating. */
	max3353_write_reg(max, MAX3353_CONTROL_2,
			  max3353_read_reg(max, MAX3353_CONTROL_2) & (~0x1));

}

/* NOTE:  three modes are possible here, only one of which
 * will be standards-conformant on any given system:
 *
 *  - OTG mode (dual-role), required if there's a Mini-AB connector
 *  - HOST mode, for when there's one or more A (host) connectors
 *  - DEVICE mode, for when there's a B/Mini-B (device) connector
 *
 * As a rule, you won't have an max3353 chip unless it's there to
 * support the OTG mode.  Other modes help testing USB controllers
 * in isolation from (full) OTG support, or maybe so later board
 * revisions can help to support those feature.
 */

#ifdef CONFIG_USB_OTG

static void
max3353_otg_enable(struct max3353 *max)
{
	u8 max_stat;
	u32 temp;

	/* Enable OTG 1ms interrupt */
	temp = MCF_USB0_OTGSC;
	temp |= OTGSC_INTR_1MS_TIMER_EN;
	temp &= ~OTGSC_CTRL_VBUS_DISCHARGE;
	MCF_USB0_OTGSC = temp;

	/* Enable MAX3353 interrupts */
	max3353_power_up(max);

	max3353_write_reg(max, MAX3353_INTERRUPT_EDGE,
			  MAX3353_INTERRUPT_EDGE_VBUS_VALID_ED |
			  MAX3353_INTERRUPT_EDGE_SESSION_VALID_ED);

	max3353_write_reg(max, MAX3353_INTERRUPT_MASK,
			  MAX3353_INTERRUPT_MASK_ID_GND_EN |
			  MAX3353_INTERRUPT_MASK_ID_FLOAT_EN |
			  MAX3353_INTERRUPT_MASK_SESSION_VALID_EN |
			  MAX3353_INTERRUPT_MASK_SESSION_END_EN |
			  MAX3353_INTERRUPT_MASK_VBUS_VALID_EN |
			  MAX3353_INTERRUPT_MASK_A_HNP_EN);

	max_stat = max3353_read_reg(max, MAX3353_STATUS);
	max->fsm.b_sess_vld = (max_stat & MAX3353_STATUS_SESSION_VALID) ? 1 : 0;
	max->fsm.a_sess_vld = max->fsm.b_sess_vld;
	max->fsm.b_sess_end = (max_stat & MAX3353_STATUS_SESSION_END) ? 1 : 0;

	max->fsm.id = (max_stat & MAX3353_STATUS_SESSION_ID_FLOAT) ? 1 : 0;
	max->otg.default_a = (max->fsm.id == 0);

}
#endif

/* add or disable the host device+driver */

static int
max3353_set_host(struct otg_transceiver *otg, struct usb_bus *host)
{
	struct max3353 *max = container_of(otg, struct max3353, otg);
	struct device *dev = host->controller;

	if (!otg || max != the_transceiver)
		return -ENODEV;

	if (!host) {
		max->otg.host = 0;
		return 0;
	}
#if !defined(CONFIG_USB_OTG) && defined(CONFIG_USB_GADGET_MCF532x)
	dev_info(&max->client.dev, "host sessions not allowed\n");
	return -EINVAL;
#else

	MCF_USB0_PORTSC = (MCF_USB0_PORTSC & ~MCF_USB_PORTSC_PTS(0xF)) | MCF_USB_PORTSC_PTS_FS_LS;	/*Set on-chip interface. */

	max->otg.host = host;

#if !defined(CONFIG_USB_GADGET_MCF532x)
	max->fsm.id = 0;
	otg->default_a = 1;
#endif

	otg->host->is_b_host = max->fsm.id;

	max->fsm.a_bus_drop = 0;
	max->fsm.a_bus_req = 1;

	otg->host->otg_port = 1;

	VDBG("host off......\n");
	if (dev && dev->driver)
		dev->driver->suspend(dev, PMSG_SUSPEND /* 0, 0 */ );
	max->host_working = 0;

#ifdef CONFIG_USB_OTG
	if (max->otg.gadget) {
		max3353_otg_enable(max);
		while (otg_fsm_statemachine(&max->fsm)) {
		};
	}
#else
	max3353_power_up(max);

	max3353_write_reg(max, MAX3353_INTERRUPT_EDGE,
			  MAX3353_INTERRUPT_EDGE_VBUS_VALID_ED |
			  MAX3353_INTERRUPT_EDGE_SESSION_VALID_ED);

	max3353_write_reg(max, MAX3353_INTERRUPT_MASK,
			  MAX3353_INTERRUPT_MASK_SESSION_VALID_EN |
			  MAX3353_INTERRUPT_MASK_SESSION_END_EN |
			  MAX3353_INTERRUPT_MASK_VBUS_VALID_EN);

	{
		u8 max_stat = max3353_read_reg(max, MAX3353_STATUS);
		max->fsm.b_sess_vld =
		    (max_stat & MAX3353_STATUS_SESSION_VALID) ? 1 : 0;
		max->fsm.a_sess_vld = max->fsm.b_sess_vld;
		max->fsm.b_sess_end =
		    (max_stat & MAX3353_STATUS_SESSION_END) ? 1 : 0;
	}

	while (otg_fsm_statemachine(&max->fsm)) {
	};
#endif

	return 0;
#endif
}

/* add or disable the peripheral device+driver */

static int
max3353_set_peripheral(struct otg_transceiver *otg, struct usb_gadget *gadget)
{
	struct max3353 *max = container_of(otg, struct max3353, otg);

#if !defined(CONFIG_USB_GADGET_MCF532x)
	dev_err(&max->client.dev, "peripheral sessions not allowed\n");
	return -EINVAL;
#else
	if (!otg || max != the_transceiver)
		return -ENODEV;

	if (!gadget) {
		if (!otg->default_a)
			otg->gadget->ops->vbus_draw(otg->gadget, 0);
		usb_gadget_vbus_disconnect(max->otg.gadget);
		max->otg.gadget = 0;
		max->fsm.b_bus_req = 0;
		otg_fsm_statemachine(&max->fsm);
		return 0;
	}
	MCF_USB0_PORTSC = (MCF_USB0_PORTSC & ~MCF_USB_PORTSC_PTS(0xF)) | MCF_USB_PORTSC_PTS_FS_LS;	/* Set on-chip interface. */

	max->otg.gadget = gadget;

#ifndef CONFIG_USB_OTG
	max->fsm.id = 1;
	otg->default_a = 0;
#endif

	max->otg.gadget->is_a_peripheral = !max->fsm.id;

	max->fsm.b_bus_req = 1;

	dev_info(&max->client.dev, "B-Peripheral sessions ok\n");

#ifdef CONFIG_USB_OTG
	if (max->otg.host) {
		max3353_otg_enable(max);
		while (otg_fsm_statemachine(&max->fsm)) {
		};
	}
#else
	max3353_power_up(max);

	max3353_write_reg(max, MAX3353_INTERRUPT_EDGE,
			  MAX3353_INTERRUPT_EDGE_SESSION_VALID_ED);

	max3353_write_reg(max, MAX3353_INTERRUPT_MASK,
			  MAX3353_INTERRUPT_MASK_SESSION_VALID_EN |
			  MAX3353_INTERRUPT_MASK_SESSION_END_EN);

	{
		u8 max_stat = max3353_read_reg(max, MAX3353_STATUS);
		max->fsm.b_sess_vld =
		    (max_stat & MAX3353_STATUS_SESSION_VALID) ? 1 : 0;
		max->fsm.a_sess_vld = max->fsm.b_sess_vld;
		max->fsm.b_sess_end =
		    (max_stat & MAX3353_STATUS_SESSION_END) ? 1 : 0;
	}

	while (otg_fsm_statemachine(&max->fsm)) {
	};

#endif

	return 0;
#endif
}

static int
max3353_detach_client(struct i2c_client *i2c)
{
	struct max3353 *max;

	max = container_of(i2c, struct max3353, client);

	max3353_power_down(max);

	/* Disable all max3353 interrupts. */
	max3353_write_reg(max, MAX3353_INTERRUPT_MASK, 0x00);

	if (max->irq > 0) {
		/*
		 * Disable interrupt request from EPORT6
		 * MCF_EPORT_EPIER&=~0x40
		 */
		MCF_EPORT_EPIER &= ~0x40;
		/*
		 * Disable interrupt request:
		 */
		MCF_INTC0_ICR(0x6) = 0x0;
		/*
		 * Set Interrupt Mask corresponding bit:
		 */
		MCF_INTC0_SIMR = 0x6;

		free_irq(max->irq, max);
	};

	MCF_USB0_OTGSC &= ~OTGSC_INTR_1MS_TIMER_EN;

	if (max->irq_otg > 0) {
		/*
		 * Disable interrupt request:
		 */
		MCF_INTC1_ICR(max->irq_otg - 128) = 0x0;
		/*
		 * Set Interrupt Mask corresponding bit:
		 */
		MCF_INTC1_SIMR = max->irq_otg - 128;

		free_irq(max->irq_otg, max);
	};

	remove_proc_file();

	put_device(&i2c->dev);
	the_transceiver = 0;

	return i2c_detach_client(i2c);
}

/* no error returns, they'd just make bus scanning stop */
static int
max3353_detect_client(struct i2c_adapter *bus, int address, int kind)
{
	int status;
	struct max3353 *max;
	struct i2c_client *i2c;

	if (the_transceiver)
		return 0;

	max = kzalloc(sizeof *max, GFP_KERNEL);
	if (!max)
		return 0;

	max->client.addr = address;
	i2c_set_clientdata(&max->client, max);
	max->client.adapter = bus;
	max->client.driver = &max3353_driver;
	strlcpy(max->client.name, DRIVER_NAME, I2C_NAME_SIZE);
	i2c = &max->client;

	/* if this is a true probe, verify the chip ... */
	if (kind < 0) {
		if ((max3353_read_reg(max, MAX3353_MANUFACTURER_0) !=
		     MAX3353_MANUFACTURER_0_VALUE) ||
		    (max3353_read_reg(max, MAX3353_MANUFACTURER_1) !=
		     MAX3353_MANUFACTURER_1_VALUE) ||
		    (max3353_read_reg(max, MAX3353_MANUFACTURER_2) !=
		     MAX3353_MANUFACTURER_2_VALUE) ||
		    (max3353_read_reg(max, MAX3353_MANUFACTURER_3) !=
		     MAX3353_MANUFACTURER_3_VALUE) ||
		    (max3353_read_reg(max, MAX3353_PRODUCT_ID_0) !=
		     MAX3353_PRODUCT_ID_0_VALUE) ||
		    (max3353_read_reg(max, MAX3353_PRODUCT_ID_1) !=
		     MAX3353_PRODUCT_ID_1_VALUE)) {
			pr_info("max3353_mcf532x i2c: addr %d is not max3353\n",
				address);
			goto fail1;
		}

	}

	status = i2c_attach_client(i2c);
	if (status < 0) {
		dev_err(&bus->dev, "can't attach %s to device %d, err %d\n",
			DRIVER_NAME, address, status);
	      fail1:
		kfree(max);
		return 0;
	}
	max->i2c_release = i2c->dev.release;
	i2c->dev.release = max3353_release;

	dev_info(&i2c->dev, "driver " DRIVER_VERSION "\n");

	/* make like power-on reset */
	max3353_write_reg(max, MAX3353_CONTROL_2, 0x1);	/* Set shut-down. */
	max3353_write_reg(max, MAX3353_CONTROL_1, 0x0);	/* Disable regs. */
	max3353_write_reg(max, MAX3353_INTERRUPT_MASK, 0x0);
	max3353_write_reg(max, MAX3353_INTERRUPT_EDGE, 0x0);
	max3353_write_reg(max, MAX3353_INTERRUPT_LATCH, 0x0);

	max->otg.dev = &max->client.dev;
	INIT_LIST_HEAD(&active_timers);
	max3353_otg_init_timers(&max->fsm);

	/* Set OTG state machine operations */
	max->fsm.ops = &max3353_otg_ops;

	/* initialize the otg structure */
	max->otg.label = DRIVER_DESC;
	max->otg.set_host = max3353_set_host;
	max->otg.set_peripheral = max3353_set_peripheral;
	max->otg.set_power = max3353_otg_set_power;
	max->otg.start_hnp = max3353_otg_start_hnp;
	max->otg.start_srp = max3353_otg_start_srp;

	max3353_power_down(max);
	the_transceiver = max;

	status = otg_set_transceiver(&max->otg);
	if (status < 0) {
		dev_err(&i2c->dev, "can't register transceiver, %d\n", status);
		goto fail2;
	}

	max->irq = 70;		/* Ext6 */
	status = request_irq(max->irq, max3353_irq,
			     SA_SAMPLE_RANDOM, DRIVER_NAME, max);
	if (status < 0) {
		dev_err(&i2c->dev, "can't get IRQ %d, err %d\n",
			max->irq, status);
	      fail2:
		i2c_detach_client(i2c);
		goto fail1;
	} else {
		/* Enable IRQ6 pin */
		MCF_GPIO_PAR_IRQ =
		    (MCF_GPIO_PAR_IRQ & ~MCF_GPIO_PAR_IRQ_PAR_IRQ6(0x3));

		/* Enable interrupt request from EPORT6 pin */
		MCF_EPORT_EPIER |= MCF_EPORT_EPIER_EPIE6;
		/*
		 * Set interrupt level:
		 */
		MCF_INTC0_ICR(6) = 0x2;
		/*
		 * Clear Interrupt Mask corresponding bit:
		 */
		MCF_INTC0_CIMR = 6;
	}

	/* Initialize the state machine structure with default values */
	SET_OTG_STATE(&max->otg, OTG_STATE_UNDEFINED);
	max->fsm.transceiver = &max->otg;

#define MCF532x_USB_OTG_INT_NUMBER		(128+47)
	max->irq_otg = MCF532x_USB_OTG_INT_NUMBER;
	status = request_irq(max->irq_otg, max3353_irq_otg,
			     SA_SHIRQ | SA_INTERRUPT, DRIVER_DESC, max);
	if (status) {
		dev_err(max->otg.dev, "can't get IRQ %d, error %d\n",
			max->irq_otg, status);
		i2c_detach_client(i2c);
		goto fail1;
	} else {
		/*
		 * Set interrupt level:
		 */
		MCF_INTC1_ICR(max->irq_otg - 128) = 0x1;
		/*
		 * Clear Interrupt Mask corresponding bit:
		 */
		MCF_INTC1_CIMR = max->irq_otg - 128;
	}

	create_proc_file();
	return 0;
}

static int
max3353_attach_adapter(struct i2c_adapter *bus)
{
	if (!i2c_check_functionality(bus, I2C_FUNC_SMBUS_BYTE_DATA
				     | I2C_FUNC_SMBUS_READ_BYTE_DATA
				     | I2C_FUNC_SMBUS_WRITE_BYTE_DATA))
		return -EINVAL;
	return i2c_probe(bus, &addr_data, max3353_detect_client);
}

static struct i2c_driver max3353_driver = {
	.driver = {
		   .name = "max3353_mcf532x",
		   },
	.id = 3353,		/* FIXME "official", i2c-ids.h */
	.class = I2C_CLASS_HWMON,
	.attach_adapter = max3353_attach_adapter,
	.detach_client = max3353_detach_client,
};

/*-------------------------------------------------------------------------*/
static int __init
max_init(void)
{
	return i2c_add_driver(&max3353_driver);
}

module_init(max_init);

static void __exit
max_exit(void)
{
	if (the_transceiver)
		otg_set_transceiver(0);
	i2c_del_driver(&max3353_driver);
}

module_exit(max_exit);

MODULE_DESCRIPTION(DRIVER_INFO);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");
