/****************************************************************************/

/*
 *	uclinux.c -- generic memory mapped MTD driver for uclinux
 *
 *	(C) Copyright 2002, Greg Ungerer (gerg@snapgear.com)
 *
 * 	$Id: uclinux.c,v 1.12 2005/11/07 11:14:29 gleixner Exp $
 */

/****************************************************************************/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/major.h>
#include <linux/root_dev.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <asm/io.h>

/****************************************************************************/

#if defined(CONFIG_M532X_MC2) || defined(CONFIG_M527X_MC2)
#include <asm/mcfsim.h>

#define FLASH_ADDR 0x00000000
#define FLASH_SIZE 0x00200000

static struct map_info m532x_flash_map = {
	.name = "MCArtistFlash",
	.size = FLASH_SIZE,
	.bankwidth = 2,
	.phys = FLASH_ADDR,
};

struct mtd_info *m532x_flash_mtdinfo;
#endif

struct map_info uclinux_ram_map = {
	.name = "RAM",
};

struct mtd_info *uclinux_ram_mtdinfo;

/****************************************************************************/

struct mtd_partition uclinux_romfs[] = {
	{ .name = "ROMfs" },
};

#define	NUM_RAM_PARTITIONS	ARRAY_SIZE(uclinux_romfs)

#if defined(CONFIG_M532X_MC2) || defined(CONFIG_M527X_MC2)
struct mtd_partition m532x_flash_partitions[] = {
	{
		.name		= "bootloader",
		.offset		= 0x00000000,
		.size		= 0x00040000,
	},
	{
		.name		= "image",
		.offset		= 0x00040000,
		.size		= 0x001C0000,
	},
	{
		.name		= "bootparms",
		.offset		= 0x00004000,
		.size		= 0x00002000,
	},
	{
		.name		= "user",
		.offset		= 0x00002000,
		.size		= 0x00002000,
	}
};
#endif

#define	NUM_FLASH_PARTITIONS	ARRAY_SIZE(m532x_flash_partitions)

/****************************************************************************/

int uclinux_point(struct mtd_info *mtd, loff_t from, size_t len,
	size_t *retlen, u_char **mtdbuf)
{
	struct map_info *map = mtd->priv;
	*mtdbuf = (u_char *) (map->virt + ((int) from));
	*retlen = len;
	return(0);
}

/****************************************************************************/

int __init uclinux_mtd_init(void)
{
	struct mtd_info *mtd;
	struct map_info *mapp;

	extern char _ebss;
	unsigned long addr = (unsigned long) &_ebss;

#if defined(CONFIG_M532X_MC2) || defined(CONFIG_M527X_MC2)
	uclinux_ram_mtdinfo = NULL;
	m532x_flash_mtdinfo = NULL;
#endif

	mapp = &uclinux_ram_map;
	mapp->phys = addr;
	mapp->size = PAGE_ALIGN(ntohl(*((unsigned long *)(addr + 8))));
	mapp->bankwidth = 4;

	printk("uclinux[mtd]: RAM probe address=0x%x size=0x%x\n",
	       	(int) mapp->phys, (int) mapp->size);

	mapp->virt = ioremap_nocache(mapp->phys, mapp->size);

	if (mapp->virt == 0) {
		printk("uclinux[mtd]: ioremap_nocache() failed\n");
		return(-EIO);
	}

	simple_map_init(mapp);

	mtd = do_map_probe("map_ram", mapp);
	if (!mtd) {
		printk("uclinux[mtd]: failed to find a mapping?\n");
		iounmap(mapp->virt);
		return(-ENXIO);
	}

	mtd->owner = THIS_MODULE;
	mtd->point = uclinux_point;
	mtd->priv = mapp;

	uclinux_ram_mtdinfo = mtd;
	add_mtd_partitions(mtd, uclinux_romfs, NUM_RAM_PARTITIONS);

	printk("uclinux[mtd]: set %s to be root filesystem\n",
	     	uclinux_romfs[0].name);
	ROOT_DEV = MKDEV(MTD_BLOCK_MAJOR, 0);

#if defined(CONFIG_M532X_MC2) || defined(CONFIG_M527X_MC2)
	// Now initialise flash ROM partitions
	mapp = &m532x_flash_map;
	printk("uclinux[mtd]: ROM probe address=0x%x size=0x%x\n",
	       	(int) mapp->phys, (int) mapp->size);

	mapp->virt = ioremap_nocache(FLASH_ADDR, FLASH_SIZE);

	simple_map_init(mapp);
	mtd = do_map_probe("cfi_probe", mapp);
	if (!mtd) {
		printk("uclinux[mtd]: failed to find a flash mapping?\n");
		iounmap(mapp->virt);
		return(-ENXIO);
	}

	mtd->owner = THIS_MODULE;
	m532x_flash_mtdinfo = mtd;

	add_mtd_partitions(mtd, m532x_flash_partitions, NUM_FLASH_PARTITIONS);
#endif

	return(0);
}

/****************************************************************************/

void __exit uclinux_mtd_cleanup(void)
{
	if (uclinux_ram_mtdinfo) {
		del_mtd_partitions(uclinux_ram_mtdinfo);
		map_destroy(uclinux_ram_mtdinfo);
		uclinux_ram_mtdinfo = NULL;
	}
	if (uclinux_ram_map.virt) {
		iounmap((void *) uclinux_ram_map.virt);
		uclinux_ram_map.virt = 0;
	}
#if defined(CONFIG_M532X_MC2) || defined(CONFIG_M527X_MC2)
	if (m532x_flash_mtdinfo) {
		del_mtd_partitions(m532x_flash_mtdinfo);
		map_destroy(m532x_flash_mtdinfo);
		m532x_flash_mtdinfo = NULL;
	}
	if (m532x_flash_map.virt) {
		iounmap((void *) m532x_flash_map.virt);
		m532x_flash_map.virt = 0;
	}
#endif
}

/****************************************************************************/

module_init(uclinux_mtd_init);
module_exit(uclinux_mtd_cleanup);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Greg Ungerer <gerg@snapgear.com>");
MODULE_DESCRIPTION("Generic RAM based MTD for uClinux");

/****************************************************************************/
