/*
 * MCF5329EVB audio driver.
 *
 * Yaroslav Vinogradov yaroslav.vinogradov@freescale.com
 * Copyright Freescale Semiconductor, Inc. 2006
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/device.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/spi/spi.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/major.h>
#include <asm/mcfsim.h>
#include <linux/interrupt.h>
#include <linux/soundcard.h>
#include <asm/uaccess.h>

#define SOUND_DEVICE_NAME "sound"

//#define AUDIO_DEBUG
#define CONFIG_AUDIODMA

#define MAX_SPEED_HZ 12000000

#define M5329_AUDIO_IRQ_SOURCE 49
#define M5329_AUDIO_IRQ_VECTOR (128+M5329_AUDIO_IRQ_SOURCE)
#define M5329_AUDIO_IRQ_LEVEL	4

/* TLV320DAC23 audio chip registers */

#define TLV320_LEFT_IN_CTL			(0x00)
#define TLV320_RIGHT_IN_CTL			(0x01)
#define TLV320_LEFT_HP_VOL_CTL		(0x02)
#define TLV320_RIGHT_HP_VOL_CTL		(0x03)
#define TLV320_ANALOG_APATH_CTL		(0x04)
#define TLV320_DIGITAL_APATH_CTL	(0x05)
#define TLV320_POWER_DOWN_CTL		(0x06)
#define TLV320_DIGITAL_IF_FMT_CTL	(0x07)
#define TLV320_SAMPLE_RATE_CTL		(0x08)
#define TLV320_DIGITAL_IF_ACT	 	(0x09)
#define TLV320_RESET_REG			(0x0f)

/* These are defines for the TLV320_SAMPLE_RATE_CTL register to
 * get the supported sample rates from the 12MHz clock used on the
 * M5329EVB.
 */
#define TLV320_SAMPLE_8KHZ		0x0C
#define TLV320_SAMPLE_16KHZ		0x58
#define TLV320_SAMPLE_22KHZ		0x62
#define TLV320_SAMPLE_32KHZ		0x18
#define TLV320_SAMPLE_44KHZ		0x22
#define TLV320_SAMPLE_48KHZ		0x00


#define	BUFSIZE		(256*1024)	/* Audio buffer data size */
#define DMASIZE		(16*1024)

#define DMA_TCD 	11

#define DMA11_IRQ_SOURCE 19
#define DMA11_IRQ_NUMBER (19+64)

struct m5329_audio {
	struct spi_device	*spi;
	u32 speed;
	u32 stereo;
	u32 bits;
	u32 format;
	u32 isopen;
	u32 txbusy;
	u32 dmaing;
	spinlock_t lock;
	u8* audio_buf;
};

static struct m5329_audio* audio_device = NULL;
volatile u32 m5329audio_start;
volatile u32 m5329audio_count;
volatile u32 m5329audio_append;
volatile u32 m5329audio_appstart;

struct m5329audio_format {
	unsigned int	format;
	unsigned int	bits;
} m5329audio_formattable[] = {
	{ AFMT_MU_LAW,		8 },
	{ AFMT_A_LAW,		8 },
	{ AFMT_IMA_ADPCM,	8 },
	{ AFMT_U8,		8 },
	{ AFMT_S16_LE,		16 },
	{ AFMT_S16_BE,		16 },
	{ AFMT_S8,		8 },
	{ AFMT_U16_LE,		16 },
	{ AFMT_U16_BE,		16 },
};

#define	FORMATSIZE	(sizeof(m5329audio_formattable) / sizeof(struct m5329audio_format))

static void m5329audio_setsamplesize(int val)
{
	int	i;

	if (audio_device == NULL) return;

	for (i = 0; (i < FORMATSIZE); i++) {
		if (m5329audio_formattable[i].format == val) {
			audio_device->format = m5329audio_formattable[i].format;
			audio_device->bits = m5329audio_formattable[i].bits;
			break;
		}
	}
}

static void m5329audio_txdrain(void)
{
#ifdef AUDIO_DEBUG
	printk("m5329audio_txdrain()\n");
#endif

	if (audio_device == NULL) return;

	while (!signal_pending(current)) {
		if (audio_device->txbusy == 0)
			break;
		current->state = TASK_INTERRUPTIBLE;
		schedule_timeout(1);
	}
}

#ifdef CONFIG_AUDIODMA

/*
 *	Configure and start DMA engine.
 */
void __inline__ m5329audio_dmarun(void)
{
#ifdef AUDIO_DEBUG
	printk("m5329audio_dmarun(): dma=%x count=%d\n",
		m5329audio_start, m5329audio_count);
#endif

/*	set_dma_mode(M5249AUDIO_DMA, DMA_MODE_WRITE|DMA_MODE_LONG_BIT);
	set_dma_device_addr(M5249AUDIO_DMA, (MCF_MBAR2+MCFA_PDOR3));
	set_dma_addr(M5249AUDIO_DMA, (int)&m5249audio_buf[m5249audio_dmastart]);
	set_dma_count(M5249AUDIO_DMA, m5249audio_dmacount);
*/

	MCF_EDMA_TCD_SADDR(DMA_TCD) = (u32)&(audio_device->audio_buf[m5329audio_start]);
	MCF_EDMA_TCD_ATTR(DMA_TCD) = MCF_EDMA_TCD_ATTR_SSIZE_32BIT | MCF_EDMA_TCD_ATTR_DSIZE_32BIT;
	MCF_EDMA_TCD_SOFF(DMA_TCD) = 4;
	MCF_EDMA_TCD_NBYTES(DMA_TCD) = 4;
	MCF_EDMA_TCD_SLAST(DMA_TCD) = 0;
	MCF_EDMA_TCD_DADDR(DMA_TCD)=(u32)&MCF_SSI_TX0;
	MCF_EDMA_TCD_CITER(DMA_TCD)= MCF_EDMA_TCD_CITER_CITER(m5329audio_count/4); //0x7FFF;
	MCF_EDMA_TCD_DOFF(DMA_TCD)=0;
	MCF_EDMA_TCD_DLAST_SGA(DMA_TCD)=0;
	MCF_EDMA_TCD_BITER(DMA_TCD)=MCF_EDMA_TCD_BITER_BITER(m5329audio_count/4);

	MCF_EDMA_SERQ = DMA_TCD;

	MCF_EDMA_TCD_CSR(DMA_TCD) |= MCF_EDMA_TCD_CSR_START | MCF_EDMA_TCD_CSR_INT_MAJOR;
	audio_device->dmaing = 1;
	audio_device->txbusy = 1;


/*	enable_dma(M5249AUDIO_DMA); */
}

/*
 *	Start DMA'ing a new buffer of data if any available.
 */
static void m5329audio_dmabuf(void)
{
#ifdef AUDIO_DEBUG
	printk("m5329audio_dmabuf(): append=%x start=%x\n", m5329audio_append, m5329audio_appstart);
#endif

	/* If already running then nothing to do... */
	if (audio_device->dmaing)
		return;

	spin_lock(&(audio_device->lock));

	/* Set DMA buffer size */
	m5329audio_count = (m5329audio_append >= m5329audio_appstart) ?
		(m5329audio_append - m5329audio_appstart) :
		(BUFSIZE - m5329audio_appstart);
	if (m5329audio_count > DMASIZE)
		m5329audio_count = DMASIZE;

	/* Adjust pointers and counters accordingly */
	m5329audio_appstart += m5329audio_count;
	if (m5329audio_appstart >= BUFSIZE)
		m5329audio_appstart = 0;

	if (m5329audio_count > 0)
		m5329audio_dmarun();
	else
		audio_device->txbusy = 0;

	spin_unlock(&(audio_device->lock));
}


/*
 *	Interrupt from DMA engine.
 */
static int m5329audio_dmaisr(int irq, void *dev_id, struct pt_regs *regs)
{
#ifdef AUDIO_DEBUG
	printk("m5329audio_dmaisr(irq=%d)\n", irq);
#endif

	spin_lock(&(audio_device->lock));

	/* Clear DMA interrupt */
	//disable_dma(M5249AUDIO_DMA);
	MCF_EDMA_CINT = DMA_TCD;
	MCF_EDMA_CERQ = DMA_TCD;

	audio_device->dmaing = 0;

	/* Update data pointers and counts */
	m5329audio_start += m5329audio_count;
	if (m5329audio_start >= BUFSIZE)
		m5329audio_start = 0;
	m5329audio_count = 0;

	spin_unlock(&(audio_device->lock));

	/* Start new DMA buffer if we can */
	m5329audio_dmabuf();

	return IRQ_HANDLED;
}

static void init_dma(void)
{
	MCF_CCM_MISCCR &= ~MCF_CCM_MISCCR_TIM_DMA; // SSI DMA Signals mapped to DMA request
	MCF_EDMA_CR = 0;

}

#endif	/* CONFIG_AUDIODMA */




/* Write CODEC register using SPI
 *   address - CODEC register address
 *   data - data to be written into register
 */
static int codec_write(u8 addr, u16 data)
{
	u16 spi_word;

	if (audio_device==NULL || audio_device->spi==NULL)
		return -ENODEV;

	spi_word = ((addr & 0x7F)<<9)|(data & 0x1FF);
	return spi_write(audio_device->spi, (const u8*)&spi_word, sizeof(spi_word));
}

static inline void enable_ssi(void)
{
	MCF_SSI_CR |= MCF_SSI_CR_SSI_EN;  // enable SSI module
	MCF_SSI_CR |= MCF_SSI_CR_TE;  // enable tranmitter
}

static inline void disable_ssi(void) {
	MCF_SSI_CR &= ~MCF_SSI_CR_TE;  // disable transmitter
	MCF_SSI_CR &= ~MCF_SSI_CR_SSI_EN;  // disable SSI module
}

/* Audio CODEC initialization */
static void init_audio_codec(void)
{
	codec_write(TLV320_RESET_REG, 0); // reset the audio chip
	udelay(500); // wait for reset
	codec_write(TLV320_POWER_DOWN_CTL, 0x001); // Turn off line input
	codec_write(TLV320_DIGITAL_IF_FMT_CTL, 0x002); // I2S slave mode
	codec_write(TLV320_DIGITAL_APATH_CTL, 0); // Set A path
	codec_write(TLV320_SAMPLE_RATE_CTL,TLV320_SAMPLE_22KHZ); // set sample rate
	codec_write(TLV320_LEFT_HP_VOL_CTL, 0x0B0); // set volume
	codec_write(TLV320_RIGHT_HP_VOL_CTL, 0x0B0); // set volume
	codec_write(TLV320_DIGITAL_IF_ACT, 1); // Activate digital interface
}

static void init_ssi(void)
{

#ifdef CONFIG_AUDIODMA
	init_dma();
#endif

	/* Enable SSI clock in CCM */
	MCF_CCM_CDR = MCF_CCM_CDR_SSIDIV(4); // 12MHz

	/* Issue a SSI reset */
	MCF_SSI_CR &= ~MCF_SSI_CR_SSI_EN;  // disable SSI module

	/* SSI module uses internal CPU clock */
	MCF_CCM_MISCCR |= MCF_CCM_MISCCR_SSI_SRC;

	MCF_SSI_CR =
			MCF_SSI_CR_I2S(1)  |  // Set I2S master mode
//			MCF_SSI_CR_I2S(2)  |  // Set I2S slave mode
			MCF_SSI_CR_MCE  |  // Set clock out on SSI_MCLK pin
			MCF_SSI_CR_SYN  |  // Enable synchronous mode
			MCF_SSI_CR_NET;
//			| MCF_SSI_CR_TCH;    // Enable two channel mode

	MCF_SSI_TCR =
			MCF_SSI_TCR_TXDIR | // int generated bit clock
			MCF_SSI_TCR_TFDIR | // int generated frame sync
			MCF_SSI_TCR_TSCKP | // Clock data on falling edge of bit clock
			MCF_SSI_TCR_TFSI   | // Frame sync active low
			MCF_SSI_TCR_TEFS  | // TX frame sync 1 bit before data
			MCF_SSI_TCR_TFSL   |
			MCF_SSI_TCR_TXBIT0 |
			MCF_SSI_TCR_TFEN0; // TX FIFO 1 enabled
//			| MCF_SSI_TCR_TFEN1;  // TX FIFO 0 enabled

	MCF_SSI_RCR =
			MCF_SSI_RCR_RSCKP | // Clock data on falling edge of bit clock
			MCF_SSI_RCR_RFSI  | // Frame sync active low
			MCF_SSI_RCR_RFSL  |
			MCF_SSI_RCR_RXBIT0 |
			MCF_SSI_RCR_REFS;   // RX frame sync 1 bit before data

	MCF_SSI_CCR = MCF_SSI_CCR_WL(7)  | // 16 bit word length
			MCF_SSI_CCR_DC(1) | // Frame rate divider
			MCF_SSI_CCR_PM(1);
//			MCF_SSI_CCR_DIV2;
//			MCF_SSI_CCR_PSR;


	MCF_SSI_IER =   // interrupts
#ifndef CONFIG_AUDIODMA
				  MCF_SSI_IER_TIE   // transmit interrupts
#else
				  MCF_SSI_IER_TDMAE
#endif
				| MCF_SSI_IER_TDE0  // transmit data register empty 0
				| MCF_SSI_IER_TFE0  // transmit FIFO 0 empty
				;

	//MCF_SSI_CR |= MCF_SSI_CR_SSI_EN;  // enable SSI module

	/* Enable the SSI pins */
	MCF_GPIO_PAR_SSI = ( 0
							| MCF_GPIO_PAR_SSI_PAR_MCLK
							| MCF_GPIO_PAR_SSI_PAR_TXD(3)
							| MCF_GPIO_PAR_SSI_PAR_RXD(3)
							| MCF_GPIO_PAR_SSI_PAR_FS(3)
							| MCF_GPIO_PAR_SSI_PAR_BCLK(3) );

	// enable IRQ
#ifndef CONFIG_AUDIODMA
	// SSI interrupt
	MCF_INTC1_ICR(M5329_AUDIO_IRQ_SOURCE) = M5329_AUDIO_IRQ_LEVEL;
	MCF_INTC1_CIMR = M5329_AUDIO_IRQ_SOURCE;
#else
	// DMA interrupt
	MCF_INTC0_ICR(DMA11_IRQ_SOURCE) = M5329_AUDIO_IRQ_LEVEL;
	MCF_INTC0_CIMR = DMA11_IRQ_SOURCE;
#endif


}

#ifndef CONFIG_AUDIODMA
static int m5329audio_isr(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned long	*bp;
	unsigned long flags;

#ifdef AUDIO_DEBUG
	//printk("m5329audio_isr(irq=%d\n", irq);
#endif

	local_irq_save(flags);

	if (m5329audio_start == m5329audio_append) {
		// *reg32p(MCFA_INTENABLE) &= ~0x00000008;
		disable_ssi();
		audio_device->txbusy = 0;
	} else {
		if (MCF_SSI_ISR & MCF_SSI_ISR_TFE0) {
			bp = (unsigned long *) &audio_device->audio_buf[m5329audio_start];
			MCF_SSI_TX0 = *bp;
			m5329audio_start += 4;
			if (m5329audio_start >= BUFSIZE)
				m5329audio_start = 0;
		}
	}
	local_irq_restore(flags);
	return IRQ_HANDLED;
}
#endif

static int m5329audio_open(struct inode *inode, struct file *filp)
{
#ifdef AUDIO_DEBUG
	printk("m5329audio_open()\n");
#endif

	if (audio_device==NULL) return (-ENODEV);

	if (audio_device->isopen)
		return(-EBUSY);

	/* Set initial driver playback defaults. */
	audio_device->speed = 44100;
	audio_device->format = AFMT_S16_BE;
	audio_device->bits = 16;
	audio_device->stereo = 1;
	audio_device->isopen = 1;
	//m5329audio_unmute();

	return(0);
}

static int m5329audio_close(struct inode *inode, struct file *filp)
{
#ifdef AUDIO_DEBUG
	printk("m5329audio_close()\n");
#endif

	if (audio_device==NULL) return (-ENODEV);

	m5329audio_txdrain();
	//m5329audio_mute();

	audio_device->txbusy = 0;
	audio_device->isopen = 0;
	m5329audio_start = 0;
	m5329audio_count = 0;
	return(0);
}

static ssize_t m5329audio_write(struct file *filp, const char *buf, size_t count, loff_t *ppos)
{
	unsigned long	*dp, *buflp;
	unsigned short	*bufwp;
	unsigned char	*bufbp;
	unsigned int	slen, bufcnt, i, s, e;

#ifdef AUDIO_DEBUG
	printk("m5329audio_write(buf=%x,count=%d)\n", (int) buf, count);
#endif

	if (audio_device==NULL) return (-ENODEV);

	if (count <= 0)
		return(0);

	buflp = (unsigned long *) buf;
	bufwp = (unsigned short *) buf;
	bufbp = (unsigned char *) buf;

	bufcnt = count & ~0x3;
	if (audio_device->stereo == 0)
		bufcnt <<= 1;
	if (audio_device->bits == 8)
		bufcnt <<= 1;

tryagain:
	/*
	 *	Get a snapshot of buffer, so we can figure out how
	 *	much data we can fit in...
	 */
	s = m5329audio_start;
	e = m5329audio_append;
	dp = (unsigned long *) &(audio_device->audio_buf[e]);

	slen = ((s > e) ? (s - e) : (BUFSIZE - (e - s))) - 4;
	if (slen > bufcnt)
		slen = bufcnt;
	if ((BUFSIZE - e) < slen)
		slen = BUFSIZE - e;

	if (slen == 0) {
		if (signal_pending(current))
			return(-ERESTARTSYS);
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(1);
		goto tryagain;
	}

	if (audio_device->stereo) {
		if (audio_device->bits == 16) {
			for (i = 0; (i < slen); i += 4)
				*dp++ = *buflp++;
		} else {
			for (i = 0; (i < slen); i += 4) {
				*dp    = (((unsigned long) *bufbp++) << 24);
				*dp++ |= (((unsigned long) *bufbp++) << 8);
			}
		}
	} else {
		if (audio_device->bits == 16) {
			for (i = 0; (i < slen); i += 4) {
				*dp++ = (((unsigned long)*bufwp)<<16) | *bufwp;
				bufwp++;
			}
		} else {
			for (i = 0; (i < slen); i += 4) {
				*dp++ = (((unsigned long) *bufbp) << 24) | 
					(((unsigned long) *bufbp) << 8);
				bufbp++;
			}
		}
	}

	e += slen;
	if (e >= BUFSIZE)
		e = 0;
	m5329audio_append = e;

	/* If not outputing audio, then start now */
	if (audio_device->txbusy == 0) {
		audio_device->txbusy++;
		//*reg32p(MCFA_INTENABLE) |= 0x00000008;
		/* Dummy write to start output interrupts */
		//*reg32p(MCFA_PDOR3) = 0;
		enable_ssi();
		m5329audio_dmabuf(); // try to start DMA
	}

	bufcnt -= slen;
	if (bufcnt > 0)
		goto tryagain;

	return(count);
}

static int m5329audio_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
	long	val;
	int	rc = 0;

#ifdef DEBUG
        printk("m5329audio_ioctl(cmd=%x,arg=%x)\n", (int) cmd, (int) arg);
#endif

	if (audio_device==NULL) return (-ENODEV);

	switch (cmd) {

	case SNDCTL_DSP_SPEED:
		rc = 0; //verify_area(VERIFY_READ, (void *) arg, sizeof(val));
		if (rc == 0) {
			get_user(val, (unsigned long *) arg);
			m5329audio_txdrain();
			audio_device->speed = val;
			/* FIXME: adjust replay speed?? */
		}
		break;

	case SNDCTL_DSP_SAMPLESIZE:
		rc = 0; //verify_area(VERIFY_READ, (void *) arg, sizeof(val));
		if (rc == 0) {
			get_user(val, (unsigned long *) arg);
			m5329audio_txdrain();
			m5329audio_setsamplesize(val);
		}
		break;

	case SNDCTL_DSP_STEREO:
		rc = 0; // verify_area(VERIFY_READ, (void *) arg, sizeof(val));
		if (rc == 0) {
			get_user(val, (unsigned long *) arg);
			m5329audio_txdrain();
			audio_device->stereo = val;
		}
		break;

	case SNDCTL_DSP_GETBLKSIZE:
		rc = 0; //verify_area(VERIFY_WRITE, (void *) arg, sizeof(long));
		if (rc == 0)
			put_user(BUFSIZE, (long *) arg);
		break;

	case SNDCTL_DSP_SYNC:
		m5329audio_txdrain();
		break;

	default:
		rc = -EINVAL;
		break;
	}

	return(rc);
}

/****************************************************************************/

struct file_operations	m5329audio_fops = {
	open: m5329audio_open,		/* open */
	release: m5329audio_close,	/* release */
	write: m5329audio_write,	/* write */
	ioctl: m5329audio_ioctl,	/* ioctl */
};

static int __devinit m5329_audio_probe(struct spi_device *spi)
{
	struct m5329_audio	*audio;
	int	err;

#ifdef AUDIO_DEBUG
	printk("audio: probe\n");
#endif

	if (!spi->irq) {
#ifdef AUDIO_D
		printk("audio: no IRQ?\n");
#endif
		dev_dbg(&spi->dev, "no IRQ?\n");
		return -ENODEV;
	}

	/* don't exceed max specified sample rate */
	if (spi->max_speed_hz > MAX_SPEED_HZ) {
		dev_dbg(&spi->dev, "f(sample) %d KHz?\n",
				(spi->max_speed_hz)/1000);
		return -EINVAL;
	}

	if (register_chrdev(SOUND_MAJOR, SOUND_DEVICE_NAME, &m5329audio_fops) < 0) {
		printk(KERN_WARNING "SOUND: failed to register major %d\n",
			SOUND_MAJOR);
		dev_dbg(&spi->dev, "SOUND: failed to register major %d\n",
			SOUND_MAJOR);
		return -ENODEV;
	}

	audio = kzalloc(sizeof(struct m5329_audio), GFP_KERNEL);
	if (!audio) {
		err = -ENOMEM;
		goto err_out;
	}

	audio->audio_buf = kmalloc(BUFSIZE, GFP_KERNEL);
	if (audio->audio_buf == NULL) {
		dev_dbg(&spi->dev,"M5329AUDIO: failed to allocate DMA[%d] buffer\n",
			BUFSIZE);
		err = -ENOMEM;
		goto err_free_mem;
	}

	audio_device = audio;

	dev_set_drvdata(&spi->dev, audio);
	spi->dev.power.power_state = PMSG_ON;

	audio->spi = spi;

#ifdef AUDIO_DEBUG
	printk("Audio IRQ=%d\n", spi->irq);
#endif

#ifndef CONFIG_AUDIODMA
	if (request_irq(spi->irq, m5329audio_isr,
			SA_INTERRUPT,	spi->dev.bus_id, audio)) {
		dev_dbg(&spi->dev, "irq %d busy?\n", spi->irq);
		err = -EBUSY;
		goto err_free_mem;
	}
#else
	if (request_irq(DMA11_IRQ_NUMBER, m5329audio_dmaisr,
			SA_INTERRUPT,	spi->dev.bus_id, audio)) {
		dev_dbg(&spi->dev, "irq %d busy?\n", DMA11_IRQ_NUMBER);
		err = -EBUSY;
		goto err_free_mem;
	}
#endif
	init_ssi();
	init_audio_codec();


#ifdef AUDIO_DEBUG
	printk("audio: Probed successfully\n");
#endif

	return 0;

 err_free_mem:
 	kfree(audio);
	audio_device = NULL;
 err_out:
 	unregister_chrdev(SOUND_MAJOR, SOUND_DEVICE_NAME);
	return err;
}

static int __devexit m5329_audio_remove(struct spi_device *spi)
{
	struct m5329_audio *audio = dev_get_drvdata(&spi->dev);
	// TODO
	m5329audio_txdrain();
#ifndef CONFIG_AUDIODMA
	free_irq(spi->irq, audio);
#else
	free_irq(DMA11_IRQ_NUMBER, audio);
#endif
	kfree(audio->audio_buf);
	kfree(audio);
	audio_device = NULL;
	unregister_chrdev(SOUND_MAJOR, SOUND_DEVICE_NAME);
	dev_dbg(&spi->dev, "unregistered audio\n");
	return 0;
}

static int m5329_audio_suspend(struct spi_device *spi, pm_message_t message) {
	return 0;
}

static int m5329_audio_resume(struct spi_device *spi) {
	return 0;
}




static struct spi_driver m5329_audio_driver = {
	.driver = {
		.name	= "m5329_audio",
		.bus	= &spi_bus_type,
		.owner	= THIS_MODULE,
	},
	.probe		= m5329_audio_probe,
	.remove		= __devexit_p(m5329_audio_remove),
	.suspend	= m5329_audio_suspend,
	.resume		= m5329_audio_resume,
};

static int __init m5329_audio_init(void)
{
	return spi_register_driver(&m5329_audio_driver);
}
module_init(m5329_audio_init);

static void __exit m5329_audio_exit(void)
{
	spi_unregister_driver(&m5329_audio_driver);
}
module_exit(m5329_audio_exit);

MODULE_DESCRIPTION("MCF5329EVB Audio Driver");
MODULE_LICENSE("GPL");
