/*
 *  linux/drivers/video/m532xfb.c -- Coldfire MCF5329 frame buffer driver
 *
 *  Copyright (c) 2006, emlix, Thomas Brinker <tb@emlix.com>
 *  Yaroslav Vinogradov yaroslav.vinogradov@freescale.com
 *  Copyright Freescale Semiconductor, Inc 2006
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Modified for Coldfire M5329 frame buffer by yaroslav.vinogradov@freescale.com
 *  Modified to new api Jan 2001 by James Simmons (jsimmons@transvirtual.com)
 *  Created 28 Dec 1997 by Geert Uytterhoeven
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/tty.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <asm/mcfsim.h>


static int nocursor = 1;

#if defined(CONFIG_LCD_800x480)
/* default modedb mode */
/* 640x480, 60 Hz, Non-Interlaced (25.172 MHz dotclock) */
static struct fb_videomode defaultmode __initdata = {
//	.refresh =	60,
	.xres =		800,
	.yres =		480,
	.pixclock =	25000,
	.left_margin =	88,
	.right_margin =	40,
	.upper_margin =	23,
	.lower_margin =	1,
	.hsync_len =	128,
	.vsync_len =	4,
	.sync =		0,
	.vmode =	FB_VMODE_NONINTERLACED
};
#endif	//defined(CONFIG_LCD_800x480)

/* Configure horizontal LCD timing
   H = H_WAIT_1 + H_WIDTH + H_WAIT_2 + XMAX
   H_WAIT1 = delay in Pixel CLK periods between OE & HSYNC      
   H_WIDTH = width of Horizontal Sync Pulse in Pixel CLK periods
   H_WAIT2 = delay between HSYNC & first data of next line */ 
 
/* Configure vertical LCD timing
   V_WAIT1 = delay between OE and VSYNC (in TFT)                       
   V_WIDTH = Vertical Sync pulse width in HSYNC periods
   V_WAIT2 = delay between VSYNC and OE of the first line of next frame*/ 

#define FB_HWAIT1 0
#define FB_HWIDTH 1
#define FB_HWAIT2 2
#define FB_VWAIT1 3
#define FB_VWIDTH 4
#define FB_VWAIT2 5

static u32  fb_wait_params[][6]  = {
/* HWAIT1, HWIDTH, HWAIT2, VWAIT1, VWIDTH, VWAIT2  */
/* 640x480 */ {48,  14, 102, 32,  1, 35},	/*10.4VGA */
/* 800x480 */ {85,  63, 148, 10,  2, 33},	/* Primeview 800x480 panel */
/* 800x600 */ {110, 59, 85,  42,  3, 24},
/* 240x320 */ {85,  10, 75,  32, 10, 10}
};

#if defined(CONFIG_LCD_640x480)
 
 #define MODE_OPTION "640x480@60"
 #define MODE_BPP 32
 #define MODE_WIDTH 640
 #define MODE_HEIGHT 480
 #define FB_WAIT_PARAMS(p) (fb_wait_params[0][(p)])
 #define PIX_CLK_DIV 3
 #define LCDC_LDCR_VALUE (MCF_LCDC_LDCR_TM(8) | MCF_LCDC_LDCR_HM(4))
 #define USE_COLOR_MAP
 
#elif defined(CONFIG_LCD_800x480)
 
 /* Coded for Primeview 800x480 panel */
 #define MODE_OPTION "800x480@60"
 #define MODE_BPP 16
 #define MODE_WIDTH 800
 #define MODE_HEIGHT 480
 #define FB_WAIT_PARAMS(p) (fb_wait_params[1][(p)])
 #define PIX_CLK_DIV 3	/* Pixel clock = 20Mhz */  
// #define PIX_CLK_DIV 2	/* Pixel clock = 26Mhz */

 #define LCDC_LDCR_VALUE (MCF_LCDC_LDCR_TM(8) | MCF_LCDC_LDCR_HM(4))

 #define USE_COLOR_MAP

#elif defined(CONFIG_LCD_800x600)

 #define MODE_OPTION "800x600@60"
 #define MODE_BPP 32 /* Default is 32 bits, 16 bits mode is also aviable */
 #define MODE_WIDTH 800
 #define MODE_HEIGHT 600
 #define FB_WAIT_PARAMS(p) (fb_wait_params[2][(p)])
 #define PIX_CLK_DIV 3
 #define LCDC_LDCR_VALUE (MCF_LCDC_LDCR_TM(8) | MCF_LCDC_LDCR_HM(4))
 #define USE_COLOR_MAP

#elif defined(CONFIG_LCD_240x320)

 #define MODE_OPTION "240x320@60"
 #define MODE_BPP 32
 #define MODE_WIDTH 240
 #define MODE_HEIGHT 320
 #define FB_WAIT_PARAMS(p) (fb_wait_params[3][(p)])
 #define PIX_CLK_DIV 3
 #define LCDC_LDCR_VALUE (MCF_LCDC_LDCR_TM(4) | MCF_LCDC_LDCR_HM(8) | MCF_LCDC_LDCR_BURST)
 #define USE_COLOR_MAP

#else
#error "LCD display resolution is not specified!"
#endif

/*
 * This structure defines the hardware state of the graphics card. Normally
 * you place this in a header file in linux/include/video. This file usually
 * also includes register information. That allows other driver subsystems
 * and userland applications the ability to use the same header file to
 * avoid duplicate work and easy porting of software.
 */
struct m532x_par {
	char mode_option[40];
/* ... */
	unsigned dump;
};

/*
 * Here we define the default structs fb_fix_screeninfo and fb_var_screeninfo
 * if we don't use modedb. If we do use modedb see xxxfb_init how to use it
 * to get a fb_var_screeninfo. Otherwise define a default var as well.
 */
static struct fb_fix_screeninfo m532xfb_fix = {
	.id =		"M532x FB",
	.type =		FB_TYPE_PACKED_PIXELS,
	.visual =	FB_VISUAL_TRUECOLOR,
	.xpanstep =	0,
	.ypanstep =	0,
	.ywrapstep =	0,
	.accel =	FB_ACCEL_NONE,
};

    /*
     * 	Modern graphical hardware not only supports pipelines but some
     *  also support multiple monitors where each display can have its
     *  its own unique data. In this case each display could be
     *  represented by a separate framebuffer device thus a separate
     *  struct fb_info. Now the struct xxx_par represents the graphics
     *  hardware state thus only one exist per card. In this case the
     *  struct xxx_par for each graphics card would be shared between
     *  every struct fb_info that represents a framebuffer on that card.
     *  This allows when one display changes it video resolution (info->var)
     *  the other displays know instantly. Each display can always be
     *  aware of the entire hardware state that affects it because they share
     *  the same xxx_par struct. The other side of the coin is multiple
     *  graphics cards that pass data around until it is finally displayed
     *  on one monitor. Such examples are the voodoo 1 cards and high end
     *  NUMA graphics servers. For this case we have a bunch of pars, each
     *  one that represents a graphics state, that belong to one struct
     *  fb_info. Their you would want to have *par point to a array of device
     *  states and have each struct fb_ops function deal with all those
     *  states. I hope this covers every possible hardware design. If not
     *  feel free to send your ideas at jsimmons@users.sf.net
     */

    /*
     *  If your driver supports multiple boards or it supports multiple
     *  framebuffers, you should make these arrays, or allocate them
     *  dynamically (using kmalloc()).
     */
static struct fb_info info;

    /*
     * Each one represents the state of the hardware. Most hardware have
     * just one hardware state. These here represent the default state(s).
     */

#define DUMP_OPTIONS 0x0 /* nothing */
//#define DUMP_OPTIONS 0xffffffff /* everything */

static struct m532x_par current_par = {
	.mode_option	= MODE_OPTION,
	.dump 		= DUMP_OPTIONS,
};

static u32 pseudo_palette[256];

int m532xfb_init(void);
int m532xfb_setup(char*);

/* ----- DUMP start ----- */

void m532xfb_dump_var(struct fb_info *info)
{
	printk("*** FB var: ***\n");
	printk("resolution: %d x %d\n", info->var.xres, info->var.yres);
	printk("virtual:    %d x %d\n", info->var.xres_virtual, info->var.yres_virtual);
	printk("offsets:    %d x %d\n", info->var.xoffset, info->var.yoffset);
	printk("bpp:        %d\n", info->var.bits_per_pixel);
	printk("grey:       %d\n", info->var.grayscale);

	printk("red:   off: %d len %d msb %d\n", info->var.red.offset, info->var.red.length, info->var.red.msb_right);
	printk("green: off: %d len %d msb %d\n", info->var.green.offset, info->var.green.length, info->var.green.msb_right);
	printk("blue:  off: %d len %d msb %d\n", info->var.blue.offset, info->var.blue.length, info->var.blue.msb_right);
	printk("transp:off: %d len %d msb %d\n", info->var.transp.offset, info->var.transp.length, info->var.transp.msb_right);

	printk("pixelformat:%d\n", info->var.nonstd);
	printk("activate:   %d\n", info->var.activate);
	printk("dimension:  %d x %d\n", info->var.height, info->var.width);

	printk("pixclock:   %lu\n", PICOS2KHZ(info->var.pixclock));
	printk("margins:    %d - %d - %d - %d\n", info->var.left_margin, info->var.right_margin, info->var.upper_margin, info->var.lower_margin);
	printk("synclen:    %d - %d\n", info->var.hsync_len, info->var.vsync_len);
	printk("sync:       %d\n", info->var.sync);
	printk("vmode:      %d\n", info->var.vmode);
	printk("rotate:     %d\n\n", info->var.rotate);
}

void m532xfb_dump_fix(struct fb_info *info)
{
	printk("*** FB fix: ***\n");
	printk("id          %s\n", info->fix.id);
	printk("smem_start  0x%08lx\n", info->fix.smem_start);
	printk("smem_len    %d\n", info->fix.smem_len);
	printk("type:       %d\n", info->fix.type);
	printk("type_aux:   %d\n", info->fix.type_aux);
	printk("visual:     %d\n", info->fix.visual);
	printk("xpanstep    %d\n", info->fix.xpanstep);
	printk("ypanstep    %d\n", info->fix.ypanstep);
	printk("ywrapstep   %d\n", info->fix.ywrapstep);
	printk("line_length %d\n", info->fix.line_length);
	printk("accel       %d\n\n", info->fix.accel);
}

void m532xfb_dump_par(struct fb_info *info)
{
	struct m532x_par *par = (struct m532x_par *) info->par;
	printk("*** FB par: ***\n");
	printk("dump:      %d\n\n", par->dump);
}

void m532xfb_dump_colors(void)
{
}

void m532xfb_dump_info(struct fb_info *info)
{
	int dump = ((struct m532x_par *) info->par)->dump;
	if (!dump) return;
	printk("-------------------------------------------------------------------\n");
	printk("*** FB info DUMP ***\n");
	printk("node:        %d\n", info->node);
	printk("flags:       %d\n\n", info->flags);
	printk("screenbase:  0x%p\n", info->screen_base);
	printk("screen_size: 0x%08lx\n", info->screen_size);
	printk("state:       %d\n\n", info->state);

	if (dump & 0x02)
		m532xfb_dump_fix(info);
	if (dump & 0x04)
		m532xfb_dump_var(info);
	if (dump & 0x08)
		m532xfb_dump_par(info);
	if (dump & 0x10)
		m532xfb_dump_colors();

    printk("*** LCD-Registers ***\n");
    printk("MCF_LCDC_LSSAR 0x%08lx\n",MCF_LCDC_LSSAR);
    printk("MCF_LCDC_LSR 0x%08lx\n",MCF_LCDC_LSR);
    printk("MCF_LCDC_LVPWR 0x%08lx\n",MCF_LCDC_LVPWR);
    printk("MCF_LCDC_LCPR 0x%08lx\n",MCF_LCDC_LCPR);
    printk("MCF_LCDC_LCWHBR 0x%08lx\n",MCF_LCDC_LCWHBR);
    printk("MCF_LCDC_LCCMR 0x%08lx\n",MCF_LCDC_LCCMR);
    printk("MCF_LCDC_LPCR 0x%08lx\n",MCF_LCDC_LPCR);
    printk("MCF_LCDC_LHCR 0x%08lx\n",MCF_LCDC_LHCR);
    printk("MCF_LCDC_LVCR 0x%08lx\n",MCF_LCDC_LVCR);
    printk("MCF_LCDC_LPOR 0x%08lx\n",MCF_LCDC_LPOR);
    printk("MCF_LCDC_LSCR 0x%08lx\n",MCF_LCDC_LSCR);
    printk("MCF_LCDC_LPCCR 0x%08lx\n",MCF_LCDC_LPCCR);
    printk("MCF_LCDC_LDCR 0x%08lx\n",MCF_LCDC_LDCR);
    printk("MCF_LCDC_LRMCR 0x%08lx\n",MCF_LCDC_LRMCR);
    printk("MCF_LCDC_LICR 0x%08lx\n",MCF_LCDC_LICR);
    printk("MCF_LCDC_LIER 0x%08lx\n",MCF_LCDC_LIER);
    printk("MCF_LCDC_LISR 0x%08lx\n",MCF_LCDC_LISR);
    printk("MCF_LCDC_LGWSAR 0x%08lx\n",MCF_LCDC_LGWSAR);
    printk("MCF_LCDC_LGWSR 0x%08lx\n",MCF_LCDC_LGWSR);
    printk("MCF_LCDC_LGWVPWR 0x%08lx\n",MCF_LCDC_LGWVPWR);
    printk("MCF_LCDC_LGWPOR 0x%08lx\n",MCF_LCDC_LGWPOR);
    printk("MCF_LCDC_LGWPR 0x%08lx\n",MCF_LCDC_LGWPR);
    printk("MCF_LCDC_LGWCR 0x%08lx\n",MCF_LCDC_LGWCR);
    printk("MCF_LCDC_LGWDCR 0x%08lx\n",MCF_LCDC_LGWDCR);
    printk("MCF_LCDC_BPLUT_BASE 0x%08lx\n",MCF_LCDC_BPLUT_BASE);
    printk("MCF_LCDC_GWLUT_BASE 0x%08lx\n",MCF_LCDC_GWLUT_BASE);
    printk("-------------------------------------------------------------------\n");
}

/* ----- DUMP end ----- */

/**
 * 
 * 
 */
static int m532xfb_blank(int blank, struct fb_info *info)
{
	//printk("m532xfb_blank");
	return 0;
}

/**
 *      m532xfb_imageblit - Copies a image from system memory to the screen. 
 *
 *      @info: frame buffer structure that represents a single frame buffer
 *	@image:	structure defining the image.
 *
 *      This drawing operation draws a image on the screen. 
 */
static void m532xfb_imageblit(struct fb_info *info, const struct fb_image *image)
{
/*
	*      @dx: The x and y coordinates of the upper left hand corner of the
	*	@dy: destination area to place the image on the screen.
	*      @width: How wide the image is we want to copy.
	*      @height: How tall the image is we want to copy.
	*      @fg_color: For mono bitmap images this is color data for     
	*      @bg_color: the foreground and background of the image to
	*		   write directly to the frmaebuffer.
	*	@depth:	How many bits represent a single pixel for this image.
	*	@data: The actual data used to construct the image on the display.
	*	@cmap: The colormap used for color images.   
 */
#if (MODE_BPP == 32)
	u32 pixelstart;
	u32 bpl = sizeof(u32);
	u32 dx = image->dx, dy = image->dy;
	u8 __iomem *dst1;
	u32 __iomem *dst;
	int i, n;
	const u8 *src = image->data;
	u32 color = 0;
	u32 *palette = NULL;
	
	//printk("m532xfb_imageblit: Image %u x %u [%u]\n",image->width, image->height, image->depth);
	//printk("ULTAN: screen_base %p\n", info->screen_base);
	
	if (image->depth == 1) 
	{
		return;
	}
	
	pixelstart = (dy * info->fix.line_length) + (dx * bpl);
	dst1 = info->screen_base + pixelstart;
	
	if (image->depth == 8)
	{	u32 pixelstart;
		u32 bpl = sizeof(u32);
		u32 dx = image->dx, dy = image->dy;
		u8 __iomem *dst1;
		u32 __iomem *dst;
		int i, n;
		const u8 *src = image->data;
		u32 color = 0;
		u32 *palette = NULL;
	
		if (image->depth == 1) 
		{
			return;
		}
	
		pixelstart = (dy * info->fix.line_length) + (dx * bpl);
		dst1 = info->screen_base + pixelstart;
	
		if (image->depth == 8)
		{
			palette = (u32 *) pseudo_palette;
		}
		 	
		for (i = image->height; i--; ) 
		{
			n = image->width;
			dst = (u32 __iomem *) dst1;
		
			while (n--) 
			{
				if (palette) 	color = palette[*src];
				else		color = *src;
			
				fb_writel(color, dst++);
			
				src++;
			}
		
			//dst1 += (info->var.xres * 4);	
			dst1 += info->fix.line_length;
		}

		palette = (u32 *) pseudo_palette;
	}
	 	
	for (i = image->height; i--; ) 
	{
		n = image->width;
		dst = (u32 __iomem *) dst1;
		
		while (n--) 
		{
			if (palette) 	color = palette[*src];
			else		color = *src;
			
			fb_writel(color, dst++);
			
			src++;
		}
		
		//dst1 += (info->var.xres * 4);	
		dst1 += info->fix.line_length;
	}
#elif (MODE_BPP == 16)

	u32 pixelstart;
	u32 bpl = sizeof(u16);
	//u32 bpp = info->var.bits_per_pixel;
	//u32 width = image->width;
	u32 dx = image->dx, dy = image->dy;
	u8 __iomem *dst1;
	u16 __iomem *dst;
	int i, n;
	const u8 *src = image->data;
	u16 color = 0;
	u16 *palette = NULL;
	
	if (image->depth == 1) 
	{
		return;
	}
	
	pixelstart = (dy * info->fix.line_length) + (dx * bpl);
	dst1 = info->screen_base + pixelstart;
	
	if (image->depth == 8)
	{
		palette = (u16 *) pseudo_palette;
	}
	 	
	for (i = image->height; i--; ) 
	{
		n = image->width;
		dst = (u16 __iomem *) dst1;
		
		while (n--) 
		{
			if (palette) 	color = (u16)palette[*src];
			else		color = *src;
			
			fb_writew(color, dst++);
			
			src++;
		}
		
		//dst1 += (info->var.xres * 4);	
		dst1 += info->fix.line_length;
	}

#endif
	
//	cfb_imageblit(info, image);
}

/**
 *      m532xfb_copyarea - Copies one area of the screen to another area.
 *
 *      @info: frame buffer structure that represents a single frame buffer
 *      @area: Structure providing the data to copy the framebuffer contents
 *	       from one region to another.
 *
 *      This drawing operation copies a rectangular area from one area of the
 *	screen to another area.
 */
static void m532xfb_copyarea(struct fb_info *info, const struct fb_copyarea *region)
{
/*
	*      @dx: The x and y coordinates of the upper left hand corner of the
	*	@dy: destination area on the screen.
	*      @width: How wide the rectangle is we want to copy.
	*      @height: How tall the rectangle is we want to copy.
	*      @sx: The x and y coordinates of the upper left hand corner of the
	*      @sy: source area on the screen.
 */
	printk("m532xfb_copyarea");
	cfb_copyarea(info, region);
}

/**
 *      m532xfb_fillrect - Draws a rectangle on the screen.		
 *
 *      @info: frame buffer structure that represents a single frame buffer
 *	@region: The structure representing the rectangular region we 
 *		 wish to draw to.
 *
 *	This drawing operation places/removes a retangle on the screen 
 *	depending on the rastering operation with the value of color which
 *	is in the current color depth format.
 */
static void m532xfb_fillrect(struct fb_info *info, const struct fb_fillrect *rect)
{
/*	Meaning of struct fb_fillrect
	*
	*	@dx: The x and y corrdinates of the upper left hand corner of the 
	*	@dy: area we want to draw to. 
	*	@width: How wide the rectangle is we want to draw.
	*	@height: How tall the rectangle is we want to draw.
	*	@color:	The color to fill in the rectangle with. 
	*	@rop: The raster operation. We can draw the rectangle with a COPY
	*	      of XOR which provides erasing effect. 
 */
	printk("m532xfb_fillrect");
	cfb_fillrect(info, rect);
}

/**
 * 
 * 
 */
static int m532xfb_cursor(struct fb_info *info, struct fb_cursor *cursor)
{
	if (nocursor)
		return 0;
	else
		return -EINVAL;		// To force soft_cursor call 
}

#ifdef USE_COLOR_MAP
/**
 *  	xxxfb_setcolreg - Optional function. Sets a color register.
 *      @regno: Which register in the CLUT we are programming
 *      @red: The red value which can be up to 16 bits wide
 *	@green: The green value which can be up to 16 bits wide
 *	@blue:  The blue value which can be up to 16 bits wide.
 *	@transp: If supported, the alpha value which can be up to 16 bits wide.
 *      @info: frame buffer info structure
 *
 *  	Set a single color register. The values supplied have a 16 bit
 *  	magnitude which needs to be scaled in this function for the hardware.
 *	Things to take into consideration are how many color registers, if
 *	any, are supported with the current color visual. With truecolor mode
 *	no color palettes are supported. Here a pseudo palette is created
 *	which we store the value in pseudo_palette in struct fb_info. For
 *	pseudocolor mode we have a limited color palette. To deal with this
 *	we can program what color is displayed for a particular pixel value.
 *	DirectColor is similar in that we can program each color field. If
 *	we have a static colormap we don't need to implement this function.
 *
 *	Returns negative errno on error, or zero on success.
 */
static int m532xfb_setcolreg(unsigned int regno, unsigned int red, unsigned int green,
			   unsigned int blue, unsigned int transp,
			   struct fb_info *info)
{
    if (regno >= 256)  /* no. of hw registers */
       return -EINVAL;
    /*
     * Program hardware... do anything you want with transp
     */

    /* grayscale works only partially under directcolor */
    if (info->var.grayscale) {
       /* grayscale = 0.30*R + 0.59*G + 0.11*B */
       red = green = blue = (red * 77 + green * 151 + blue * 28) >> 8;
    }

    //printk(KERN_INFO "m532xfb_setcolreg: Reg %u [R %02X G %02X B %02X]\n", regno, red, green, blue);
    
    /* Directcolor:
     *   var->{color}.offset contains start of bitfield
     *   var->{color}.length contains length of bitfield
     *   {hardwarespecific} contains width of DAC
     *   cmap[X] is programmed to (X << red.offset) | (X << green.offset) | (X << blue.offset)
     *   RAMDAC[X] is programmed to (red, green, blue)
     *
     * Pseudocolor:
     *    uses offset = 0 && length = DAC register width.
     *    var->{color}.offset is 0
     *    var->{color}.length contains widht of DAC
     *    cmap is not used
     *    DAC[X] is programmed to (red, green, blue)
     * Truecolor:
     *    does not use RAMDAC (usually has 3 of them).
     *    var->{color}.offset contains start of bitfield
     *    var->{color}.length contains length of bitfield
     *    cmap is programmed to (red << red.offset) | (green << green.offset) |
     *                      (blue << blue.offset) | (transp << transp.offset)
     *    RAMDAC does not exist
     */
#define CNVT_TOHW(val,width) ((((val)<<(width))+0x7FFF-(val))>>16)
    switch (info->fix.visual) {
       case FB_VISUAL_TRUECOLOR:
       case FB_VISUAL_PSEUDOCOLOR:
               red = CNVT_TOHW(red, info->var.red.length);
               green = CNVT_TOHW(green, info->var.green.length);
               blue = CNVT_TOHW(blue, info->var.blue.length);
               transp = CNVT_TOHW(transp, info->var.transp.length);
               break;
       case FB_VISUAL_DIRECTCOLOR:
	       /* example here assumes 8 bit DAC. Might be different
		* for your hardware */
               red = CNVT_TOHW(red, 8);
               green = CNVT_TOHW(green, 8);
               blue = CNVT_TOHW(blue, 8);
               /* hey, there is bug in transp handling... */
               transp = CNVT_TOHW(transp, 8);
               break;
    }
#undef CNVT_TOHW
    /* Truecolor has hardware independent palette */
    if (info->fix.visual == FB_VISUAL_TRUECOLOR) {
       u32 v;

       if (regno >= 16)
           return -EINVAL;
      
       v = (red << info->var.red.offset) |
           (green << info->var.green.offset) |
           (blue << info->var.blue.offset) |
           (transp << info->var.transp.offset);

       switch (info->var.bits_per_pixel) {
		case 8:
			/* Yes some hand held devices have this. */
           		((u8*)(info->pseudo_palette))[regno] = v;
			break;
   		case 16:
           		((u16*)(info->pseudo_palette))[regno] = v;
			break;
		case 24:
		case 32:((u32*)(info->pseudo_palette))[regno] = v;
			break;
       }
       return 0;
    }
    /* ... */
    //printk("do something with color palette!\n");
    
    return 0;
}
#endif //USE_COLOR_MAP

#if defined(CONFIG_M532X_MC2)
static int m532xfb_ioctl(struct fb_info *info, unsigned int cmd,
			    unsigned long arg)
{
	switch(cmd) {
	   case FBIO_SHOW_LOGO:
		if (fb_prepare_logo(info, FB_ROTATE_UR)) {
			fb_show_logo(info, FB_ROTATE_UR);
		}
		break;
	   default:
		return -EINVAL;
	}
	return 0;
}
#endif

/**
 * Detect old revision of M5329 EVB
 * 
 */
int detect_old_rev(void)
{
#if defined(CONFIG_M532X_MC2)
	return 0;
#else
	unsigned char podr, pddr, par;
	int retval;

	par = MCF_GPIO_PAR_FEC;
	pddr = MCF_GPIO_PDDR_FECH;
	podr = MCF_GPIO_PODR_FECH;

	/* Determine LogicPC PCB revision by writing a 1 to FEC_TXD0.
	   A pull down on this line causes new boards to read back a 0.
	   A floating line on the old boards causes a 1 to be detected. */
	MCF_GPIO_PAR_FEC &= 0x03;
	MCF_GPIO_PDDR_FECH |= MCF_GPIO_PDDR_FECH_PDDR_FECH5;
	MCF_GPIO_PODR_FECH |= MCF_GPIO_PODR_FECH_PODR_FECH5;
	MCF_GPIO_PDDR_FECH &= ~MCF_GPIO_PDDR_FECH_PDDR_FECH5;
	udelay(500);
	if(MCF_GPIO_PPDSDR_FECH & MCF_GPIO_PPDSDR_FECH_PPDSDR_FECH5)
		retval = 1; /* Older revision detected */
	else
		retval = 0;

	printk(KERN_INFO "M5329EVB %s Board Revision\n", (retval) ? "Old" : "New");
	
	MCF_GPIO_PAR_FEC = par;
	MCF_GPIO_PDDR_FECH = pddr;
	MCF_GPIO_PODR_FECH = podr;
	
	return(retval);
#endif
}

/**
 * Initialise M532x LCD controller
 * 
 */
static void m532xfb_LcdControllerInit(void)
{
	// Ensure that LCD controller clock is disabled before making any changes
	MCF_CCM_MISCCR &= ~MCF_CCM_MISCCR_LCD_CHEN;
			
	MCF_LCDC_LSSAR  = (unsigned int)info.screen_base;
	MCF_LCDC_LSR    = MCF_LCDC_LSR_XMAX(MODE_WIDTH/16) 
			| MCF_LCDC_LSR_YMAX(MODE_HEIGHT);
	
	{
		int words_per_row = MODE_WIDTH >> 2;
#if (MODE_BPP == 32)
		words_per_row <<= 2;	//18bpp
#elif (MODE_BPP == 16)
		words_per_row <<= 1;	//16bpp
#endif
		MCF_LCDC_LVPWR = words_per_row;
	}
	

	if(detect_old_rev()){
#if defined(CONFIG_LCD_640x480)
		MCF_LCDC_LPCR   = MCF_LCDC_LPCR_MODE_TFT
				| MCF_LCDC_LPCR_BPIX_18bpp
				| MCF_LCDC_LPCR_FLM
				| MCF_LCDC_LPCR_LPPOL
				| MCF_LCDC_LPCR_OEPOL
				| MCF_LCDC_LPCR_CLKPOL
				| MCF_LCDC_LPCR_SCLKSEL
				| MCF_LCDC_LPCR_ACDSEL
				| MCF_LCDC_LPCR_ENDSEL
				| MCF_LCDC_LPCR_PCD(PIX_CLK_DIV);
#elif defined(CONFIG_LCD_800x480)
		MCF_LCDC_LPCR   = MCF_LCDC_LPCR_MODE_TFT
	#if MODE_BPP == 32
				| MCF_LCDC_LPCR_BPIX_18bpp
	#else
				| MCF_LCDC_LPCR_BPIX_16bpp
	#endif
				| MCF_LCDC_LPCR_FLM		// First line marker, Active low
				| MCF_LCDC_LPCR_LPPOL		// Line pulse polarity, Active low
				| MCF_LCDC_LPCR_CLKPOL		// LCD Shift clock polarity
				| MCF_LCDC_LPCR_OEPOL		// Enable LCD_SCLK
				//| MCF_LCDC_LPCR_SCLKIDLE	// SCLK power saving idle
				| MCF_LCDC_LPCR_ENDSEL		// Memory image data, 0 = little endian, 1 = big endian
				//| MCF_LCDC_LPCR_SWAP_SEL	// Byte swapping control
				//| MCF_LCDC_LPCR_REV_VS	// Vertical scan direction
				//| MCF_LCDC_LPCR_ACDSEL 	// ACD Clock Source select
				//| MCF_LCDC_LPCR_ACD(ACD_DIV)  // ACD Stuff
				| MCF_LCDC_LPCR_SCLKSEL 	// LCD_SCLK Select, 0 = Disable OE & LSCLK when no data output in TFT mode for power savings, 1 = Always on
				|MCF_LCDC_LPCR_PCD(PIX_CLK_DIV); // Pixel CLK Divider, actual divider is PCD value + 
#elif defined(CONFIG_LCD_800x600)
		MCF_LCDC_LPCR   = MCF_LCDC_LPCR_MODE_TFT
	#if MODE_BPP == 32
				| MCF_LCDC_LPCR_BPIX_18bpp
	#else
				| MCF_LCDC_LPCR_BPIX_16bpp
	#endif
				| MCF_LCDC_LPCR_FLM
				| MCF_LCDC_LPCR_LPPOL
				| MCF_LCDC_LPCR_CLKPOL
				| MCF_LCDC_LPCR_OEPOL
				| MCF_LCDC_LPCR_ACDSEL
				| MCF_LCDC_LPCR_SCLKSEL
				| MCF_LCDC_LPCR_ENDSEL
				| MCF_LCDC_LPCR_PCD(PIX_CLK_DIV);
#elif defined(CONFIG_LCD_240x320)
		MCF_LCDC_LPCR   = MCF_LCDC_LPCR_TFT
				| MCF_LCDC_LPCR_COLOR
				| MCF_LCDC_LPCR_BPIX_18bpp
				| MCF_LCDC_LPCR_FLM
				| MCF_LCDC_LPCR_LPPOL
				| MCF_LCDC_LPCR_OEPOL
				| MCF_LCDC_LPCR_CLKPOL
				| MCF_LCDC_LPCR_SCLKSEL
				| MCF_LCDC_LPCR_ACDSEL
				| MCF_LCDC_LPCR_ENDSEL
				| MCF_LCDC_LPCR_PCD(PIX_CLK_DIV);
#endif
	}
	else
	{
#if !defined(CONFIG_M532X_MC2)
		/* Setup contrast register */
		MCF_LCDC_LPCCR = 0x00000510;
#endif
#if defined(CONFIG_LCD_640x480)
		MCF_LCDC_LPCR      = MCF_LCDC_LPCR_TFT
				| MCF_LCDC_LPCR_COLOR
				| MCF_LCDC_LPCR_BPIX_18bpp
				| MCF_LCDC_LPCR_FLM
				| MCF_LCDC_LPCR_LPPOL
				| MCF_LCDC_LPCR_CLKPOL
				| MCF_LCDC_LPCR_SCLKSEL
				| MCF_LCDC_LPCR_ACDSEL
				| MCF_LCDC_LPCR_ENDSEL
				| MCF_LCDC_LPCR_PCD(PIX_CLK_DIV);
#elif defined(CONFIG_LCD_800x480)
		MCF_LCDC_LPCR   = MCF_LCDC_LPCR_TFT
				| MCF_LCDC_LPCR_COLOR
	#if (MODE_BPP == 32)
				| MCF_LCDC_LPCR_BPIX_18bpp
				| MCF_LCDC_LPCR_SWAP_SEL	// Byte swapping control
	#else
				| MCF_LCDC_LPCR_BPIX_16bpp
				| MCF_LCDC_LPCR_ENDSEL		// Memory image data, 0 = little endian, 1 = big endian (needed to get pixels in the right place)
		// BUT TEST MODE RECTANGLE RIGHT HAND LINE IS MISSING!!!
	#endif
				| MCF_LCDC_LPCR_FLM		// First line marker, Active low
				| MCF_LCDC_LPCR_LPPOL		// Line pulse polarity, Active low
				| MCF_LCDC_LPCR_CLKPOL		// LCD Shift clock polarity
				| MCF_LCDC_LPCR_SCLKSEL 	// LCD_SCLK Select, 0 = Disable OE & LSCLK when no data output in TFT mode for power savings, 1 = Always on
				|MCF_LCDC_LPCR_PCD(PIX_CLK_DIV); // Pixel CLK Divider, actual divider is PCD value + 1 
#elif defined(CONFIG_LCD_800x600)
		MCF_LCDC_LPCR      = MCF_LCDC_LPCR_MODE_TFT
	#if MODE_BPP == 32
				| MCF_LCDC_LPCR_BPIX_18bpp
	#else
				| MCF_LCDC_LPCR_BPIX_16bpp
	#endif
				| MCF_LCDC_LPCR_FLM
				| MCF_LCDC_LPCR_LPPOL
				| MCF_LCDC_LPCR_CLKPOL
				| MCF_LCDC_LPCR_ACDSEL
				| MCF_LCDC_LPCR_SCLKSEL
				| MCF_LCDC_LPCR_ENDSEL
				| MCF_LCDC_LPCR_PCD(PIX_CLK_DIV);
#elif defined(CONFIG_LCD_240x320)
		MCF_LCDC_LPCR      = MCF_LCDC_LPCR_TFT
				| MCF_LCDC_LPCR_COLOR
				| MCF_LCDC_LPCR_BPIX_18bpp
				| MCF_LCDC_LPCR_FLM
				| MCF_LCDC_LPCR_LPPOL
				| MCF_LCDC_LPCR_CLKPOL
				| MCF_LCDC_LPCR_SCLKSEL
				| MCF_LCDC_LPCR_ACDSEL
				| MCF_LCDC_LPCR_ENDSEL
				| MCF_LCDC_LPCR_PCD(PIX_CLK_DIV);
#endif
	}

	MCF_LCDC_LHCR = MCF_LCDC_LHCR_H_WIDTH(FB_WAIT_PARAMS(FB_HWIDTH))
			| MCF_LCDC_LHCR_H_WAIT_1(FB_WAIT_PARAMS(FB_HWAIT1))
			| MCF_LCDC_LHCR_H_WAIT_2(FB_WAIT_PARAMS(FB_HWAIT2));

	MCF_LCDC_LVCR = MCF_LCDC_LVCR_V_WIDTH(FB_WAIT_PARAMS(FB_VWIDTH))
			| MCF_LCDC_LVCR_V_WAIT_1(FB_WAIT_PARAMS(FB_VWAIT1))
			| MCF_LCDC_LVCR_V_WAIT_2(FB_WAIT_PARAMS(FB_VWAIT2));

	MCF_LCDC_LPOR   = MCF_LCDC_LPOR_POS(0);
	MCF_LCDC_LDCR   = LCDC_LDCR_VALUE;

	/* connect ldc controller to clock */
	MCF_CCM_MISCCR  |= MCF_CCM_MISCCR_LCD_CHEN;
}

/* ------------------------------------------------------------------------- */

/*
 *  Frame buffer operations
 */

static struct fb_ops m532xfb_ops = {
	.owner		= THIS_MODULE,
	.fb_fillrect	= m532xfb_fillrect, 	
	.fb_copyarea	= m532xfb_copyarea,		
	.fb_imageblit	= m532xfb_imageblit,	
	.fb_cursor	= m532xfb_cursor,
	.fb_blank       = m532xfb_blank,
#ifdef USE_COLOR_MAP
	.fb_setcolreg	= m532xfb_setcolreg,
#endif
#if defined(CONFIG_M532X_MC2)
	.fb_ioctl	= m532xfb_ioctl
#endif
};

/*
 *  Initialization
 */

int __init m532xfb_init(void)
{
	int cmap_len=256, retval;
#ifndef CONFIG_M532X_MC2
	char* mode_option=NULL;
#endif
	/*
	*  For kernel boot options (in 'video=xxxfb:<options>' format)
	*/
#ifndef MODULE
	char *option = NULL;

	if (fb_get_options("m532xfb", &option)) {
		printk("No fb on command line specified\n");
		return -ENODEV;
	}
	m532xfb_setup(option);
#endif

	printk("Initing M532x Framebuffer\n");

	info.fbops = &m532xfb_ops;
	info.fix = m532xfb_fix;
	info.pseudo_palette = pseudo_palette;

	/*
	* Set up flags to indicate what sort of acceleration your
	* driver can provide (pan/wrap/copyarea/etc.) and whether it
	* is a module -- see FBINFO_* in include/linux/fb.h
	*/
	info.flags = FBINFO_DEFAULT + FBINFO_HWACCEL_DISABLED;
	info.par = &current_par;

#if defined(CONFIG_M532X_MC2)
	retval = fb_find_mode(&info.var, &info, NULL, NULL, 0, &defaultmode, MODE_BPP);
#else
	/*
	* This should give a reasonable default video mode. The following is
	* done when we can set a video mode.
	*/
	if (!mode_option)
		mode_option = MODE_OPTION;
	
	retval = fb_find_mode(&info.var, &info, mode_option, NULL, 0, NULL, MODE_BPP);
#endif

	if (!retval || retval == 4)
		return -EINVAL;

	info.screen_size = (info.var.xres * info.var.yres * info.var.bits_per_pixel) / 8;
	info.var.xres_virtual = info.var.xres;
	info.var.yres_virtual = info.var.yres;

#if MODE_BPP == 32
	info.var.red.offset = 18;
	info.var.red.length = 6;
	info.var.red.msb_right = 0;

	info.var.green.offset = 10;
	info.var.green.length = 6;
	info.var.green.msb_right = 0;

	info.var.blue.offset = 2;
	info.var.blue.length = 6;
	info.var.blue.msb_right = 0;

	info.var.transp.offset = 0;
	info.var.transp.length = 0;
	info.var.transp.msb_right = 0;
#else
	info.var.red.offset = 11;
	info.var.red.length = 5;

	info.var.green.offset = 5;
	info.var.green.length = 6;

	info.var.blue.offset = 0;
	info.var.blue.length = 5;
#endif

	/*
	* Here we set the screen_base to the virtual memory address
	* for the framebuffer. Usually we obtain the resource address
	* from the bus layer and then translate it to virtual memory
	* space via ioremap. Consult ioport.h.
	*/
	info.screen_base = (unsigned char *)__get_free_pages(GFP_KERNEL, get_order(info.screen_size));
	if (!info.screen_base) {
		printk("Unable to allocate %d PAGEs(%ld Bytes) fb memory\n",get_order(info.screen_size),info.screen_size);
		return -ENOMEM;
	}

	info.fix.smem_start = virt_to_phys((void *)info.screen_base);
	info.fix.smem_len = info.screen_size;
	info.fix.line_length = info.var.xres * info.var.bits_per_pixel / 8;

	/*
	* Set page reserved so that mmap will work. This is necessary
	* since we'll be remapping normal memory.
	*/
	{
		unsigned char * page;
		for (page = info.screen_base;
			(unsigned long)page < (PAGE_ALIGN((unsigned long)info.screen_base + info.screen_size));
			page += PAGE_SIZE)
		{
			SetPageReserved(virt_to_page(page));
		}
	}
	memset((void *)info.screen_base, 0, info.screen_size);

	/* set gpios */
	MCF_GPIO_PAR_LCDDATA = 0xff; /* switch all to display */
#if defined(CONFIG_M532X_MC2)
	//MCF_GPIO_PAR_LCDCTL  = 0x0f9;	/* PS, ACD and REV pins configured for GPIO */
	#ifdef TFT_MOD_OE
	MCF_GPIO_PAR_LCDCTL  = 0x1e9;		/* CONTRAST, PS and REV pins configured for GPIO */	
	#else
	MCF_GPIO_PAR_LCDCTL  = 0x0e9;		/* CONTRAST, PS, ACD and REV pins configured for GPIO */
	#endif

	/* Set LCD PS (LCDCTLL2) and CONTRAST (LCDCTLL6) pins as output */
	MCF_GPIO_PDDR_LCDCTLL  |= (MCF_GPIO_PDDR_LCDCTLL_PDDR_LCDCTLL2 | MCF_GPIO_PDDR_LCDCTLL_PDDR_LCDCTLL6);		
	
	MCF_GPIO_PODR_LCDCTLL &= ~0x04;	/* LCD_PS line low to turn on backlight */
	#ifndef TFT_MOD_OE
	MCF_GPIO_PODR_LCDCTLL &= ~0x40;	/* LCD_CONTRAST line low to setr DENB low on timing controller */
	#endif
#else
	MCF_GPIO_PAR_LCDCTL  = 0x1ff;
#endif
	
#if defined(CONFIG_M532X_MC2)
	/* Adjust crossbar priorities to give LCD adequate bandwidth
	   (Slave port 1 = flex bus and SDRAM) */

	#if 0
	MCF_XBS_PRS1 =    MCF_XBS_PRIO_CORE(MCF_PRIO_LVL_1)
			| MCF_XBS_PRIO_FEC(MCF_PRIO_LVL_2)
			| MCF_XBS_PRIO_LCD(MCF_PRIO_LVL_3)
			| MCF_XBS_PRIO_EDMA(MCF_PRIO_LVL_4)
			| MCF_XBS_PRIO_USBHOST(MCF_PRIO_LVL_5)
			| MCF_XBS_PRIO_USBOTG(MCF_PRIO_LVL_6)
			| MCF_XBS_PRIO_FACTTEST(MCF_PRIO_LVL_7);

	//MCF_XBS_CRS1 = MCF_XBS_ARB_ROTATING;		// Round-robin priority  .... Makes ths one worse
	#endif
	#if 1
	MCF_XBS_PRS1 =    MCF_XBS_PRIO_FEC(MCF_PRIO_LVL_1)
			| MCF_XBS_PRIO_LCD(MCF_PRIO_LVL_2)
			| MCF_XBS_PRIO_CORE(MCF_PRIO_LVL_3)
			| MCF_XBS_PRIO_EDMA(MCF_PRIO_LVL_4)
			| MCF_XBS_PRIO_USBHOST(MCF_PRIO_LVL_5)
			| MCF_XBS_PRIO_USBOTG(MCF_PRIO_LVL_6)
			| MCF_XBS_PRIO_FACTTEST(MCF_PRIO_LVL_7);
	
	//MCF_XBS_CRS1 = MCF_XBS_ARB_ROTATING;		// Round-robin priority  .... Makes ths one worse
	#endif
	#if 0	//Almost, but not quite!
	MCF_XBS_PRS1 =    MCF_XBS_PRIO_LCD(MCF_PRIO_LVL_1)
			| MCF_XBS_PRIO_FEC(MCF_PRIO_LVL_2)
			| MCF_XBS_PRIO_CORE(MCF_PRIO_LVL_3)
			| MCF_XBS_PRIO_EDMA(MCF_PRIO_LVL_4)
			| MCF_XBS_PRIO_USBHOST(MCF_PRIO_LVL_5)
			| MCF_XBS_PRIO_USBOTG(MCF_PRIO_LVL_6)
			| MCF_XBS_PRIO_FACTTEST(MCF_PRIO_LVL_7);
	
	//MCF_XBS_CRS1 = MCF_XBS_ARB_ROTATING;		// Round-robin priority  .... Makes ths one worse
	#endif
			
	/* burst mode */
	MCF_SCM_BCR = MCF_SCM_BCR_GBR | MCF_SCM_BCR_GBW | 0xFF;
#else
	/* burst mode */
	MCF_SCM_BCR = 0x3ff;
#endif

	/* Initialise LCD controller */
	m532xfb_LcdControllerInit();
	
	/* This has to been done !!! */
	fb_alloc_cmap(&info.cmap, cmap_len, 0);

	/*
	* The following is done in the case of having hardware with a static
	* mode. If we are setting the mode ourselves we don't call this.
	*/
    if (register_framebuffer(&info) < 0)
		return -EINVAL;
	printk(KERN_INFO "fb%d: %s frame buffer device\n", info.node, info.fix.id);

    m532xfb_dump_info(&info);

    return 0;
}

/*
*  Cleanup
*/
static void __exit m532xfb_cleanup(void)
{
	/*
	*  If your driver supports multiple boards, you should unregister and
	*  clean up all instances.
	*/
    unregister_framebuffer(&info);
    fb_dealloc_cmap(&info.cmap);
    /* ... */
}

/*
*  Setup
*/

/*
 * Only necessary if your driver takes special options,
 * otherwise we fall back on the generic fb_setup().
 */
int __init m532xfb_setup(char *options)
{
    /* Parse user speficied options (`video=xxxfb:') */
    return 0;
}
/* ------------------------------------------------------------------------- */
/*
*  Modularization
*/
module_init(m532xfb_init);
module_exit(m532xfb_cleanup);

MODULE_AUTHOR("Thomas Brinker <tb@emlix.com>");
MODULE_DESCRIPTION("MCF532x Framebuffer");
MODULE_LICENSE("GPL");
