/***************************************************************************/
/*                  Copyright (c) 2007 Dark Matter Digital Inc             */
/***************************************************************************/

// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// See <http://www.gnu.org/copyleft/lesser.html>
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.

/***************************************************************************
                  INCLUDE 
 ***************************************************************************/
//#define DEBUG

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <pthread.h>

#include "avahi-common/malloc.h"
#include "avahi-common/simple-watch.h"
#include "avahi-common/alternative.h"
#include "avahi-common/timeval.h"

#include "avahi-core/core.h"
#include "avahi-core/log.h"
#include "avahi-core/publish.h"
#include "avahi-core/lookup.h"
#include "avahi-core/dns-srv-rr.h"

#include "avahi_helper.h"

/***************************************************************************
                  EXTERNALS 
 ***************************************************************************/
 
/***************************************************************************
                  DEFINITIONS 
 ***************************************************************************/

// Define the following to enable debug logging from Avahi library
//#define	AVAHI_DEBUG_LOG		1

//  Define the following to use only user name from persistent storage for Discovery service name 
// [Note: This was requested to be neater for host software but was removed as host software currently 
//  	  needs to see unit type name in service name].
//#define AVAHI_USER_NAME_ONLY	1

#define LOCAL static

#ifdef DEBUG
#define debug(x)	printf x
#else
#define debug(x)
#endif

#define addptr(p,o)    ((void *)(((unsigned char*)(p))+((unsigned long)(o))))
	
/***************************************************************************
                  STRUCTURES 
 ***************************************************************************/

/***************************************************************************
                  DATA 
 ***************************************************************************/

LOCAL AvahiSimplePoll 	*pSimple_poll;
LOCAL const AvahiPoll 	*pAvahiPoll_API;

LOCAL AvahiSEntryGroup  *pAvahiGroup;

LOCAL AvahiServer 	*pAvahiServer;
LOCAL char 		*pAvahiServiceName;

LOCAL int		ServiceNextUnitNumber;

LOCAL AVAHI_HELPER_SHARED *pAvahiHelperShared;

/***************************************************************************
                  DECLARATIONS
 ***************************************************************************/

LOCAL int AvahiCreateEntries(const char *hostName, const char *nMACAddr);
LOCAL void AvahiEntryGroupCallbackFunction(AVAHI_GCC_UNUSED AvahiServer *s, AVAHI_GCC_UNUSED AvahiSEntryGroup *g, AvahiEntryGroupState state, AVAHI_GCC_UNUSED void* userdata); 

/***************************************************************************
                  CODE 
 ***************************************************************************/

#ifdef AVAHI_DEBUG_LOG
/**
 * Callback function for logging Avahi output
 *
 * @return 
 */
void AvahiDebugFunction(AvahiLogLevel level, const char *txt)
{
	printf("AVAHI: %s\n", txt);
}
#endif

/**
 * 
 *
 * @return 
 */
LOCAL void AvahiRemoveEntries(void) 
{
    	if (pAvahiGroup)
	{
        	avahi_s_entry_group_reset(pAvahiGroup);
	}
}

/**
 * Create service name with incrementing number for uniqueness
 *
 * @return 
 */
LOCAL char *AvahiCreateServiceName(char *buffer, int num)
{
	const char *unitTypeName;
	const char *unitName;
	char  unitnum[12];

	assert(pAvahiHelperShared);
	assert(pAvahiHelperShared->UnitTypeName);

	unitTypeName = pAvahiHelperShared->UnitTypeName;
	unitName = pAvahiHelperShared->UnitUserName;
	
#ifdef AVAHI_USER_NAME_ONLY		// Don't include unit type in name if user has set a name
	if (unitName && (unitName[0] != '\0'))
	{
		strcpy(buffer, unitName);
	}
	else
	{
		strcpy(buffer, unitTypeName);
	}

	if (num > 0)
	{
		char unitnum[12];

		// Append number to create unique name
		sprintf(unitnum, " - %u", num);
		strcat(buffer, unitnum);
	}
#else
	strcpy(buffer, unitTypeName);

	if (unitName && (unitName[0] != '\0'))
	{
		strcat(buffer, " (");
		strcat(buffer, unitName);
		if (num > 0)
		{
			// Append number to create unique name
			sprintf(unitnum, " %u", num);
			strcat(buffer, unitnum);
		}
		strcat(buffer, ")");
	}
	else if (num > 0)
	{
		char unitnum[12];

		// Append number to create unique name
		sprintf(unitnum, " - %u", num);
		strcat(buffer, unitnum);
	}
#endif

	return buffer;
}

/**
 * 
 *
 * @return 
 */
LOCAL int AvahiCreateEntries(const char *hostRecord, const char *addrRecord) 
{
	AvahiStringList *strlst = NULL;
	char 	serviceNameBuffer[255];
	int 	result = 0;

	debug(("AvahiCreateEntries\n"));

	assert(pAvahiHelperShared);

	AvahiRemoveEntries();

	if (!pAvahiGroup) 
        {
		pAvahiGroup = avahi_s_entry_group_new(pAvahiServer, AvahiEntryGroupCallbackFunction, NULL);
	}

	assert(avahi_s_entry_group_is_empty(pAvahiGroup));

	pAvahiServiceName = avahi_strdup(AvahiCreateServiceName(serviceNameBuffer, ServiceNextUnitNumber));

	// Create TXT record containing local ethernet address	
	strlst = avahi_string_list_add(strlst, pAvahiHelperShared->TextRecordLocalMAC);

	// Create additional TXT records if required
	if (hostRecord) 
	{
		strlst = avahi_string_list_add(strlst, hostRecord);
	}

	if (addrRecord)
	{
		strlst = avahi_string_list_add(strlst, addrRecord);
	}

	debug(("Avahi ServiceType %s\n", pAvahiHelperShared->ServiceType));
	debug(("Avahi ServicePort %u\n", pAvahiHelperShared->ServicePort));

	// Apply records to service
	result = avahi_server_add_service_strlst(pAvahiServer, pAvahiGroup, AVAHI_IF_UNSPEC, AVAHI_PROTO_UNSPEC, 0, \
				 		 pAvahiServiceName, pAvahiHelperShared->ServiceType, \
						 NULL, NULL, pAvahiHelperShared->ServicePort, strlst);

	if (result < 0)  
	{
		avahi_log_error("Failed to add service records");
		if (pAvahiGroup) 
        	{
			avahi_s_entry_group_free(pAvahiGroup);
			pAvahiGroup = NULL;
		}
		if (strlst)
		{
			avahi_string_list_free(strlst);
		}		
		return result;
	}

	avahi_s_entry_group_commit(pAvahiGroup);

	if (strlst)
	{
		avahi_string_list_free(strlst);
	}

	return result;
}

/**
 * 
 *
 * @return 
 */
LOCAL void AvahiEntryGroupCallbackFunction(AVAHI_GCC_UNUSED AvahiServer *s, AVAHI_GCC_UNUSED AvahiSEntryGroup *g, AvahiEntryGroupState state, AVAHI_GCC_UNUSED void* userdata) 
{
	//avahi_log_debug("entry group state: %i", state); 

	if (state == AVAHI_ENTRY_GROUP_COLLISION) 
	{
        	avahi_log_debug("Service name conflict name <%s>", pAvahiServiceName);
		ServiceNextUnitNumber++;
		AvahiRemoveEntries();
		AvahiCreateEntries(NULL, NULL);
    	} 
	else if (state == AVAHI_ENTRY_GROUP_ESTABLISHED) 
	{
        	avahi_log_debug("Service established under name <%s>", pAvahiServiceName);
    	}
}

/**
 * 
 *
 * @return 
 */
LOCAL void AvahiServerCallbackFunction(AvahiServer *s, AvahiServerState state, AVAHI_GCC_UNUSED void* userdata) 
{
	pAvahiServer = s;
    
    	//avahi_log_debug("AvahiServerCallbackFunction, server state: %i", state); 
    
    	if (state == AVAHI_SERVER_RUNNING)
	{
        	avahi_log_debug("Avahi Server startup complete.\nHost name is <%s>. Service cookie is %u", avahi_server_get_host_name_fqdn(s), avahi_server_get_local_service_cookie(s));
        	(void) AvahiCreateEntries(NULL, NULL);
   	} 
	else if (state == AVAHI_SERVER_COLLISION) 
	{
		char *n;
		AvahiRemoveEntries();

		n = avahi_alternative_host_name(avahi_server_get_host_name(s));

		avahi_log_debug("Host name conflict, retrying with <%s>", n);
		avahi_server_set_host_name(s, n);
		avahi_free(n);
    }
}

/**
 * 
 *
 * @return 
 */
LOCAL int AvahiHelperInit(AVAHI_MSG_INIT *pMsg)
{
    	AvahiServerConfig config;
	int error = 0;

	debug(("AvahiHelper: Init\n"));

	pAvahiHelperShared = pMsg->pSharedContext;

	pSimple_poll = NULL;
	pAvahiServer = NULL;
	pAvahiServiceName = NULL;
	pAvahiGroup = NULL;
	
	ServiceNextUnitNumber = 0;

#ifdef AVAHI_DEBUG_LOG
	avahi_set_log_function(AvahiDebugFunction);
#endif

	pSimple_poll = avahi_simple_poll_new();
	pAvahiPoll_API = avahi_simple_poll_get(pSimple_poll);
   
	avahi_server_config_init(&config);
 
	config.publish_hinfo = 0;
    	config.publish_workstation = 0;
 	
	pAvahiServer = avahi_server_new(pAvahiPoll_API, &config, AvahiServerCallbackFunction, NULL, &error);
   	avahi_server_config_free(&config);

	return error;
}

/**
 * 
 *
 * @return 
 */
LOCAL void AvahiHelperClose()
{
	debug(("AvahiHelper: Close\n"));

  	if (pAvahiGroup)
	{
        	avahi_s_entry_group_free(pAvahiGroup);
		pAvahiGroup = NULL; 
	}

	if (pSimple_poll)
	{
        	avahi_simple_poll_free(pSimple_poll);
		pSimple_poll = NULL;
	}

	if (pAvahiServer)
	{
		avahi_server_free(pAvahiServer);
		pAvahiServer = NULL;
	}

	if (pAvahiServiceName)
	{
		avahi_free(pAvahiServiceName);
		pAvahiServiceName = NULL;	
	}
}

/**
 * 
 *
 * @return 
 */
LOCAL int AvahiHelperService(AVAHI_MSG_SERVICE *pMsg)
{
	int r = 0;	
	int timeout_msec = pMsg->Timeout;

	if (pSimple_poll)
	{
		r = avahi_simple_poll_iterate(pSimple_poll, timeout_msec);
		if (r<0) printf("Avahi Poll Error %d\n", r);
	}

	return r;
}

/**
 * Add optional host info TXT records to Avahi service 
 *
 * @return 
 */
LOCAL int AvahiHelperUpdateRecords(void)
{
	int r = 0;	

	debug(("AvahiHelper: UpdateRecords\n"));

	r = AvahiCreateEntries(pAvahiHelperShared->TextRecordHost, pAvahiHelperShared->TextRecordHostMAC);

	return r;
}

/**
 * Remove host info TXT records from Avahi service on connection reset
 *
 * @return 
 */
LOCAL int AvahiHelperReset(void)
{
	int result = 0;

	debug(("AvahiHelperReset\n"));

	// Remove host info TXT records from Avahi service
	result = AvahiCreateEntries(NULL, NULL);	

	// Allow service name to be updated again	
	ServiceNextUnitNumber = 0;

	return result; 
}

/**
 * 
 *
 * @return 
 */
LOCAL int ProcessServerMessage(AVAHI_MSG_HDR *pMsgHdr)
{
	int shutdown = 0;
	int result = 0;

	switch (pMsgHdr->Type)
	{
		case MSG_AVAHI_INIT:
				result = AvahiHelperInit((AVAHI_MSG_INIT*)pMsgHdr);
				break;
		case MSG_AVAHI_CLOSE:
				AvahiHelperClose();
				shutdown = 1;
				break;
		case MSG_AVAHI_SERVICE:	
				result = AvahiHelperService((AVAHI_MSG_SERVICE*)pMsgHdr);
				break;
		case MSG_AVAHI_RESET:
				result = AvahiHelperReset();
				break;
		case MSG_AVAHI_UPDATE_RECORDS:
				result = AvahiHelperUpdateRecords();
				break;
		default:	printf("AvahiHelper: Unknown Message\n");
				break;
	}

	assert(pAvahiHelperShared);

	pAvahiHelperShared->MessageResult = result;
	pAvahiHelperShared->MessageComplete = 1;

	return shutdown;
}
	
/**
 * 
 *
 * @return 
 */
LOCAL void ProcessServerMessages(int client_socket_fd)
{
	unsigned char msgBuffer[512];
	AVAHI_MSG_HDR *pMsgHdr = (AVAHI_MSG_HDR *)&msgBuffer;
	int shutdown = 0;
	int nAdditionalBytes;

	while (!shutdown)
	{
		if (read(client_socket_fd, pMsgHdr, sizeof(AVAHI_MSG_HDR)) == 0)
		{
			// Client closed connection
			return;
		}

		assert(pMsgHdr->Length >= sizeof(AVAHI_MSG_HDR));

		nAdditionalBytes = pMsgHdr->Length - sizeof(AVAHI_MSG_HDR);

		if (nAdditionalBytes)
		{
			read(client_socket_fd, addptr(pMsgHdr, sizeof(AVAHI_MSG_HDR)), nAdditionalBytes);
		}

		shutdown = ProcessServerMessage(pMsgHdr);		 
	}
}

/**
 * Application entry point 
 * 
 * @return 
 */
int main () 
{
	debug(("Starting Avahi Helper..\n"));

	int socket_fd;
	struct sockaddr_un name;

	struct sockaddr_un client_name;
	socklen_t clinet_name_len;
	int client_socket_fd;

	pAvahiHelperShared = NULL;

	/* Create socket */
	socket_fd = socket(PF_LOCAL, SOCK_STREAM, 0);

	/* Indicate that this is a server */
	name.sun_family = AF_LOCAL;
	strcpy(name.sun_path, SOCKET_AVAHI_NAME);
	if (bind(socket_fd, (struct sockaddr *)&name, SUN_LEN(&name)) < 0)
	{
		printf("AvahiHelper: Bind failed\n");
	}

	/* Listen for connection */
	if (listen(socket_fd, 5) != 0)
	{
		printf("AvahiHelper: listen failed\n");
	}

	client_socket_fd = accept(socket_fd, (struct sockaddr *)&client_name, &clinet_name_len);
	debug(("AvahiHelper: Accepted connection\n"));
	
	ProcessServerMessages(client_socket_fd);

	close(client_socket_fd);

	/* Remove the socket file */
	close(socket_fd);
	unlink(SOCKET_AVAHI_NAME);
	
	exit(0);
}

