/*
 * $Id: m5253_flash.c,v 1.0 2006/11/02 $
 *
 * Handle mapping of the flash on MCF5253EVB boards
 *
 * Author:	Hongjun,Chen <hong-jun.chen@freescale.com>
 * Copyright:	(C) 2006 Freescale Semiconductor Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>

/*
 * MCF5253EVB flash layout (2M bytes)
 * No   Start           End             Size            Content
 * 1    0xffe00000      0xffe2ffff      0x30000         dBUG image
 * 2    0xffe30000      0xffe3ffff      0x10000         Real-time clock 
 * 							 date information
 * 3    0xffe40000      0xffffffff      0x1c0000        uClinux Kernel image
 */

#define WINDOW_ADDR 0xffe00000
#define WINDOW_SIZE 0x00200000

/* Flash / Partition sizing */
#define FLASH_MAX_SIZE          0x200000
#define PARTITION0_ADDR         0x00
#define PARTITION0_SIZE         0x30000
#define PARTITION1_ADDR         0x30000
#define PARTITION1_SIZE         0x10000
#define PARTITION2_ADDR         0x40000
#define PARTITION2_SIZE         0x1c0000

/* partition_info gives details on the logical partitions that the split the
 * single flash device into. If the size if zero we use up to the end of the
 * device. 
 */
static struct mtd_partition partition_info[]={
	{
		.name		= "dBUG partition",
		.offset 	= PARTITION0_ADDR,
		.size		= PARTITION0_SIZE,
	},	
	{
		.name		= "Date partition",
		.offset		= PARTITION1_ADDR,
		.size		= PARTITION1_SIZE
	},
	{
		.name		= "Kernel partition",
		.offset		= PARTITION2_ADDR,
		.size		= PARTITION2_SIZE
	}
};

#define PARTITION_NUM (sizeof(partition_info)/sizeof(struct mtd_partition))

static struct mtd_info *mymtd;

struct map_info m5253_map = {
	.name		= "MCF5253EVB Flash Map Info",
	.size		= WINDOW_SIZE,
	.phys		= WINDOW_ADDR,
	.bankwidth	= 2,
};

int __init init_m5253(void)
{
	printk(KERN_NOTICE "MCF5253EVB flash device: %x at %x Partition number %d\n",
			WINDOW_SIZE, WINDOW_ADDR, PARTITION_NUM);
	m5253_map.virt = ioremap(WINDOW_ADDR, WINDOW_SIZE);

	if (!m5253_map.virt) {
		printk("Failed to ioremap\n");
		return -EIO;
	}

	simple_map_init(&m5253_map);
	
	mymtd = do_map_probe("cfi_probe", &m5253_map);
	if (mymtd) {
		mymtd->owner = THIS_MODULE;
                add_mtd_partitions(mymtd, partition_info, PARTITION_NUM);
		printk(KERN_NOTICE "MCF5253EVB flash device initialized\n");
		return 0;
	}

	iounmap((void *)m5253_map.virt);
	return -ENXIO;
}

static void __exit cleanup_m5253(void)
{
	if (mymtd) {
		del_mtd_device(mymtd);
		map_destroy(mymtd);
	}
	if (m5253_map.virt) {
		iounmap((void *)m5253_map.virt);
		m5253_map.virt = 0;
	}
}

module_init(init_m5253);
module_exit(cleanup_m5253);

MODULE_AUTHOR("Hongjun,Chen <hong-jun.chen@freescale.com>");
MODULE_DESCRIPTION("MTD map driver for Freescale MCF5253EVB board");
MODULE_LICENSE("Dual BSD/GPL");
