/*
 * driver/usb/gadget/mcf532x_udc.c
 *
 * MCF532x USB Device Controller Driver
 *
 * Andrey Butok
 * Copyright Freescale Semiconductor, Inc. 2006
 *
 *      Based on mpc_udc.c code
 *      of Li Yang and Jiang Bo (Freescale Semiconductor, Inc.)
 *
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 ***************************************************************************
 * Changes:
 *   v0.003	27 October 2006		Andrey Butok
 *		Added M5329EVB support (no external transceiver).
 *   v0.002	29 September 2006	Andrey Butok
 *		Some little changes. Added OTG support.
 *   v0.001	12 July 2006		Andrey Butok
 *   		Initial Release - developed on uClinux with 2.6.17.1 kernel.
 *   		Based on mpc_udc.c code
 *   		of Li Yang and Jiang Bo (Freescale Semiconductor, Inc.)
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/mm.h>
#include <linux/moduleparam.h>
#include <linux/device.h>
#include <linux/usb_ch9.h>
#include <linux/usb_gadget.h>
#include <linux/usb_otg.h>
#include <linux/platform_device.h>
#include <linux/pm.h>

#include <asm/byteorder.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/unaligned.h>

#include <asm/m532xsim.h>
#include <asm/cacheflush.h>

#include "mcf532x_udc.h"

#undef	USB_TRACE

#define	DRIVER_DESC	"Freescale MCF532x UDC driver"
#define	DRIVER_AUTHOR	"Andrey Butok, Freescale Semiconductor Inc."
#define	DRIVER_VERSION	"27 October 2006"

#define	DMA_ADDR_INVALID	(~(dma_addr_t)0)

static const char driver_name[] = "mcf532x_udc";
static const char driver_desc[] = DRIVER_DESC;

/* it is initialized in probe()  */
static struct mcf532x_udc *udc_controller = NULL;

/* ep_qh_base store the base address before 2K align */
static struct ep_queue_head *ep_qh_base = NULL;

/*ep name is important in gadget, it should obey the convention of ep_match()*/
static const char *const ep_name[] = {
	"ep0-control", NULL,	/* everyone has ep0 */
	/* 5 configurable endpoints */
	"ep1out",
	"ep1in",
	"ep2out",
	"ep2in",
	"ep3out",
	"ep3in"
};
static struct usb_endpoint_descriptor
 mcf532x_ep0_desc = {
	.bLength = USB_DT_ENDPOINT_SIZE,
	.bDescriptorType = USB_DT_ENDPOINT,
	.bEndpointAddress = 0,
	.bmAttributes = USB_ENDPOINT_XFER_CONTROL,
	.wMaxPacketSize = USB_MAX_CTRL_PAYLOAD,
};

static int mcf532x_udc_suspend(struct device *dev, pm_message_t state);
static int mcf532x_udc_resume(struct device *dev);

/********************************************************************
 * 	Internal Used Function
********************************************************************/
/*-----------------------------------------------------------------
 * done() - retire a request; caller blocked irqs
 * @status : when req->req.status is -EINPROGRESSS, it is input para
 *	     else it will be a output parameter
 * req->req.status : in ep_queue() it will be set as -EINPROGRESS
 *--------------------------------------------------------------*/
static void
done(struct mcf532x_ep *ep, struct mcf532x_req *req, int status)
{
	struct mcf532x_udc *udc = NULL;
	unsigned char stopped = ep->stopped;

	udc = (struct mcf532x_udc *) ep->udc;
	/* the req->queue pointer is used by ep_queue() func, in which
	 * the request will be added into a udc_ep->queue 'd tail
	 * so here the req will be dropped from the ep->queue
	 */
	list_del_init(&req->queue);

	/* req.status should be set as -EINPROGRESS in ep_queue() */
	if (req->req.status == -EINPROGRESS)
		req->req.status = status;
	else
		status = req->req.status;

	if (status && (status != -ESHUTDOWN))
		VDBG("complete %s req %p stat %d len %u/%u",
		     ep->ep.name, &req->req, status,
		     req->req.actual, req->req.length);

	/* don't modify queue heads during completion callback */
	ep->stopped = 1;

	spin_unlock(&ep->udc->lock);
	/* this complete() should a func implemented by gadget layer, 
	 * eg fsg->bulk_in_complete() */
	if (req->req.complete)
		req->req.complete(&ep->ep, &req->req);

	spin_lock(&ep->udc->lock);
	ep->stopped = stopped;
}

/*----------------------------------------------------------------- 
 * nuke(): delete all requests related to this ep
 * called by ep_disable() within spinlock held 
 * add status paramter?
 *--------------------------------------------------------------*/
static void
nuke(struct mcf532x_ep *ep, int status)
{
	ep->stopped = 1;

	/* Whether this eq has request linked */
	while (!list_empty(&ep->queue)) {
		struct mcf532x_req *req = NULL;

		req = list_entry(ep->queue.next, struct mcf532x_req, queue);

		done(ep, req, status);
	}
}

/*------------------------------------------------------------------
	Internal Hardware related function
 ------------------------------------------------------------------*/

/* @qh_addr is the aligned virt addr of ep QH addr
 * it is used to set endpointlistaddr Reg */
static int
dr_controller_setup(void *qh_addr)
{
	unsigned int tmp = 0;
	int timeout;

	/* before here, make sure usb_slave_regs has been initialized */
	if (!qh_addr)
		return -EINVAL;

	/* Stop and reset the usb controller */
	MCF_USB0_USBCMD &= ~MCF_USB_USBCMD_RS;

	MCF_USB0_USBCMD = MCF_USB_USBCMD_RST;

	/* Wait reset completed */
	timeout = 500;
	while (timeout-- && (MCF_USB0_USBCMD & 0x2))
		udelay(1);
	if (timeout <= 0)
		ERR("%s - USBCMD_RST never clear. Timeout is %d \n",
		    __FUNCTION__, timeout);
	/*
	 * Set "Device mode","Big endian byte ordering" and "Setup lockouts off":
	 */
	MCF_USB0_USBMODE = MCF_USB_USBMODE_SLOM |
	    MCF_USB_USBMODE_ES | MCF_USB_USBMODE_CM_DEVICE;

	/* Clear the setup status */
	MCF_USB0_USBSTS = 0;

	tmp = virt_to_phys(qh_addr);
	tmp &= MCF_USB_EPLISTADDR_EPBASE(0xFFFFFFFF);
	MCF_USB0_EPLISTADDR = tmp;

	VDBG("vir[qh_base] is %p phy[qh_base] is 0x%8x reg is 0x%8x",
	     qh_addr, (int) tmp, MCF_USB0_EPLISTADDR);

	return 0;
}

static int
can_pullup(struct mcf532x_udc *udc)
{
	return udc->driver && udc->softconnect && udc->vbus_active;
}

static void
pullup_enable(struct mcf532x_udc *udc)
{
	u32 ccm = MCF_CCM_UOCSR;
	ccm |= MCF_CCM_UOCSR_BVLD;
	ccm &= ~MCF_CCM_UOCSR_SEND;
	MCF_CCM_UOCSR = ccm;

	MCF_USB0_USBINTR = MCF_USB_USBINTR_UE |
	    MCF_USB_USBINTR_UEE |
	    MCF_USB_USBINTR_PCE |
	    MCF_USB_USBINTR_URE | MCF_USB_USBINTR_SLE | MCF_USB_USBINTR_SEE;

	MCF_USB0_USBCMD |= MCF_USB_USBCMD_RS;	/* Run */

	if(!udc->transceiver)
		*((u16 *)(MCF_FBCS1_CSAR)) &= ~0x1; /* Enable pullup register */
}

static void
pullup_disable(struct mcf532x_udc *udc)
{
	/* disable all INTRs */
	MCF_USB0_USBINTR = 0;

	MCF_CCM_UOCSR &= ~MCF_CCM_UOCSR_BVLD;

	MCF_USB0_USBCMD &= ~MCF_USB_USBCMD_RS;	/* Stop */

	if(!udc->transceiver)
		*((u16 *)(MCF_FBCS1_CSAR)) |= 0x1; /* Disable pullup register */

}

static void
dr_controller_run(struct mcf532x_udc *udc)
{
	pullup_enable(udc);
	udc->stopped = 0;
}

static void
dr_controller_stop(struct mcf532x_udc *udc)
{
	pullup_disable(udc);
	udc->stopped = 1;
	udc->gadget.b_hnp_enable = 0;
	udc->gadget.a_hnp_support = 0;
	udc->gadget.a_alt_hnp_support = 0;
}

void
dr_ep_setup(unsigned char ep_num, unsigned char dir, unsigned char ep_type)
{
	unsigned int tmp_epctrl = 0;

	tmp_epctrl = MCF_USB0_EPCR(ep_num);
	if (dir) {
		if (ep_num)
			tmp_epctrl |= MCF_USB_EPCR_TXR;
		tmp_epctrl |= MCF_USB_EPCR_TXE;
		tmp_epctrl |= MCF_USB_EPCR_TXT(ep_type);
	} else {
		if (ep_num)
			tmp_epctrl |= MCF_USB_EPCR_RXR;
		tmp_epctrl |= MCF_USB_EPCR_RXE;
		tmp_epctrl |= MCF_USB_EPCR_RXT(ep_type);
	}

	MCF_USB0_EPCR(ep_num) = tmp_epctrl;

	/* wait the write reg finished */
	while (!(MCF_USB0_EPCR(ep_num) &
		 (tmp_epctrl & (MCF_USB_EPCR_TXE | MCF_USB_EPCR_RXE)))) ;
}

static void
dr_ep_change_stall(unsigned char ep_num, unsigned char dir, int value)
{
	unsigned int tmp_epctrl = 0;

	tmp_epctrl = MCF_USB0_EPCR(ep_num);

	if (value) {
		/* set the stall bit */
		if (dir)
			tmp_epctrl |= MCF_USB_EPCR_TXS;
		else
			tmp_epctrl |= MCF_USB_EPCR_RXS;
	} else {
		/* clear the stall bit and reset data toggle */
		if (dir) {
			tmp_epctrl &= ~MCF_USB_EPCR_TXS;
			tmp_epctrl |= MCF_USB_EPCR_TXR;
		} else {
			tmp_epctrl &= ~MCF_USB_EPCR_RXS;
			tmp_epctrl |= MCF_USB_EPCR_RXR;
		}

	}

	MCF_USB0_EPCR(ep_num) = tmp_epctrl;
}

/********************************************************************
	Internal Structure Build up functions 
********************************************************************/

/*------------------------------------------------------------------
* struct_ep_qh_setup(): set the Endpoint Capabilites field of QH 
 * @zlt: Zero Length Termination Select 
 * @mult: Mult field 
 ------------------------------------------------------------------*/
static void
struct_ep_qh_setup(void *handle, unsigned char ep_num,
		   unsigned char dir, unsigned char ep_type,
		   unsigned int max_pkt_len,
		   unsigned int zlt, unsigned char mult)
{
	struct mcf532x_udc *udc = NULL;
	struct ep_queue_head *p_QH = NULL;
	unsigned int tmp = 0;

	udc = (struct mcf532x_udc *) handle;

	p_QH = &udc->ep_qh[2 * ep_num + dir];

	/* set the Endpoint Capabilites Reg of QH */
	switch (ep_type) {
	case USB_ENDPOINT_XFER_CONTROL:
		/* Interrupt On Setup (IOS). for control ep  */
		tmp = (max_pkt_len << EP_QUEUE_HEAD_MAX_PKT_LEN_POS) |
		    EP_QUEUE_HEAD_IOS;
		break;
	case USB_ENDPOINT_XFER_ISOC:
		tmp = (max_pkt_len << EP_QUEUE_HEAD_MAX_PKT_LEN_POS) |
		    (mult << EP_QUEUE_HEAD_MULT_POS);
		break;
	case USB_ENDPOINT_XFER_BULK:
	case USB_ENDPOINT_XFER_INT:
		tmp = max_pkt_len << EP_QUEUE_HEAD_MAX_PKT_LEN_POS;
		if (zlt)
			tmp |= EP_QUEUE_HEAD_ZLT_SEL;
		break;
	default:
		VDBG("error ep type is %d", ep_type);
		return;
	}
	p_QH->max_pkt_length = tmp;
	flush_dcache_range((unsigned long) p_QH, (unsigned long) p_QH +
			   sizeof (struct ep_queue_head));

	return;
}

/* This function only to make code looks good
 * it is a collection of struct_ep_qh_setup and dr_ep_setup for ep0
 * ep0 should set OK before the bind() of gadget layer
 */
static void
ep0_dr_and_qh_setup(struct mcf532x_udc *udc)
{
	/* the intialization of an ep includes: fields in QH, Regs,
	 * mcf532x_ep struct */
	struct_ep_qh_setup(udc, 0, USB_RECV,
			   USB_ENDPOINT_XFER_CONTROL, USB_MAX_CTRL_PAYLOAD,
			   0, 0);
	struct_ep_qh_setup(udc, 0, USB_SEND,
			   USB_ENDPOINT_XFER_CONTROL, USB_MAX_CTRL_PAYLOAD,
			   0, 0);
	dr_ep_setup(0, USB_RECV, USB_ENDPOINT_XFER_CONTROL);
	dr_ep_setup(0, USB_SEND, USB_ENDPOINT_XFER_CONTROL);

	return;
}

/***********************************************************************
		Endpoint Management Functions
***********************************************************************/

/*-------------------------------------------------------------------------
 * when configurations are set, or when interface settings change
 * for example the do_set_interface() in gadget layer,
 * the driver will enable or disable the relevant endpoints
 * ep0 will not use this func it is enable in probe()
-------------------------------------------------------------------------*/
static int
mcf532x_ep_enable(struct usb_ep *_ep,
		  const struct usb_endpoint_descriptor *desc)
{
	struct mcf532x_udc *udc = NULL;
	struct mcf532x_ep *ep = NULL;
	unsigned short max = 0;
	unsigned char mult = 0, zlt = 0;
	int retval = 0;
	unsigned long flags = 0;
	char *val = NULL;	/* for debug */

	ep = container_of(_ep, struct mcf532x_ep, ep);

	/* catch various bogus parameters */
	if (!_ep || !desc || ep->desc || _ep->name == ep_name[0] ||
	    (desc->bDescriptorType != USB_DT_ENDPOINT))
		/* FIXME: add judge for ep->bEndpointAddress */
		return -EINVAL;

	udc = ep->udc;

	if (!udc->driver || (udc->gadget.speed == USB_SPEED_UNKNOWN))
		return -ESHUTDOWN;

	max = __constant_le16_to_cpu(desc->wMaxPacketSize);
	retval = -EINVAL;

	/* check the max package size validate for this endpoint */
	/* Refer to USB2.0 spec table 9-13, 
	 */
	switch (desc->bmAttributes & 0x03) {
	case USB_ENDPOINT_XFER_BULK:
		if (strstr(ep->ep.name, "-iso")
		    || strstr(ep->ep.name, "-int"))
			goto en_done;
		mult = 0;
		zlt = 1;
		switch (udc->gadget.speed) {
		case USB_SPEED_HIGH:
			if ((max == 128) || (max == 256) || (max == 512))
				break;
		default:
			switch (max) {
			case 4:
			case 8:
			case 16:
			case 32:
			case 64:
				break;
			default:
			case USB_SPEED_LOW:
				goto en_done;
			}
		}
		break;
	case USB_ENDPOINT_XFER_INT:
		if (strstr(ep->ep.name, "-iso"))	/* bulk is ok */
			goto en_done;
		mult = 0;
		zlt = 1;
		switch (udc->gadget.speed) {
		case USB_SPEED_HIGH:
			if (max <= 1024)
				break;
		case USB_SPEED_FULL:
			if (max <= 64)
				break;
		default:
			if (max <= 8)
				break;
			goto en_done;
		}
		break;
	case USB_ENDPOINT_XFER_ISOC:
		if (strstr(ep->ep.name, "-bulk") || strstr(ep->ep.name, "-int"))
			goto en_done;
		mult =
		    (unsigned char) (1 +
				     ((__constant_le16_to_cpu
				       (desc->wMaxPacketSize)
				       >> 11) & 0x03));
		zlt = 0;
		switch (udc->gadget.speed) {
		case USB_SPEED_HIGH:
			if (max <= 1024)
				break;
		case USB_SPEED_FULL:
			if (max <= 1023)
				break;
		default:
			goto en_done;
		}
		break;
	case USB_ENDPOINT_XFER_CONTROL:
		if (strstr(ep->ep.name, "-iso")
		    || strstr(ep->ep.name, "-int"))
			goto en_done;
		mult = 0;
		zlt = 1;
		switch (udc->gadget.speed) {
		case USB_SPEED_HIGH:
		case USB_SPEED_FULL:
			switch (max) {
			case 1:
			case 2:
			case 4:
			case 8:
			case 16:
			case 32:
			case 64:
				break;
			default:
				goto en_done;
			}
		case USB_SPEED_LOW:
			switch (max) {
			case 1:
			case 2:
			case 4:
			case 8:
				break;
			default:
				goto en_done;
			}
		default:
			goto en_done;
		}
		break;

	default:
		goto en_done;
	}

	/* here initialize variable of ep */
	spin_lock_irqsave(&udc->lock, flags);
	ep->ep.maxpacket = max;
	ep->desc = desc;
	ep->stopped = 0;

	/* hardware special operation */

	/* Init EPx Queue Head (Ep Capabilites field in QH 
	 * according to max, zlt, mult) */
	struct_ep_qh_setup((void *) udc, (unsigned char) ep_index(ep),
			   (unsigned char) ((desc->
					     bEndpointAddress & USB_DIR_IN) ?
					    USB_SEND : USB_RECV),
			   (unsigned char) (desc->bmAttributes &
					    USB_ENDPOINT_XFERTYPE_MASK),
			   max, zlt, mult);

	/* Init endpoint x at here */
	dr_ep_setup((unsigned char) ep_index(ep),
		    (unsigned char) ((desc->bEndpointAddress & USB_DIR_IN)
				     ? USB_SEND : USB_RECV),
		    (unsigned char) (desc->bmAttributes &
				     USB_ENDPOINT_XFERTYPE_MASK));

	/* Now HW will be NAKing transfers to that EP, 
	 * until a buffer is queued to it. */

	/* should have stop the lock */
	spin_unlock_irqrestore(&udc->lock, flags);
	retval = 0;
	switch (desc->bmAttributes & 0x03) {
	case USB_ENDPOINT_XFER_BULK:
		val = "bulk";
		break;
	case USB_ENDPOINT_XFER_ISOC:
		val = "iso";
		break;
	case USB_ENDPOINT_XFER_INT:
		val = "intr";
		break;
	default:
		val = "ctrl";
		break;
	}

	VDBG("enabled %s (ep%d%s-%s) maxpacket %d", ep->ep.name,
	     ep->desc->bEndpointAddress & 0x0f,
	     (desc->bEndpointAddress & USB_DIR_IN) ? "in" : "out", val, max);
      en_done:
	return retval;
}

/*---------------------------------------------------------------------
 * @ep : the ep being unconfigured. May not be ep0
 * Any pending and uncomplete req will complete with status (-ESHUTDOWN)
*---------------------------------------------------------------------*/
static int
mcf532x_ep_disable(struct usb_ep *_ep)
{
	struct mcf532x_udc *udc = NULL;
	struct mcf532x_ep *ep = NULL;
	unsigned long flags = 0;

	ep = container_of(_ep, struct mcf532x_ep, ep);
	if (!_ep || !ep->desc) {
		VDBG("%s not enabled", _ep ? ep->ep.name : NULL);
		return -EINVAL;
	}

	udc = (struct mcf532x_udc *) ep->udc;

	spin_lock_irqsave(&udc->lock, flags);

	/* Nuke all pending requests (does flush) */
	nuke(ep, -ESHUTDOWN);

	ep->desc = 0;
	ep->stopped = 1;
	spin_unlock_irqrestore(&udc->lock, flags);

	VDBG("disabled %s OK", _ep->name);
	return 0;
}

/*---------------------------------------------------------------------
 * allocate a request object used by this endpoint 
 * the main operation is to insert the req->queue to the eq->queue
 * Returns the request, or null if one could not be allocated
*---------------------------------------------------------------------*/
static struct usb_request *
mcf532x_alloc_request(struct usb_ep *_ep, gfp_t gfp_flags)
{
	struct mcf532x_req *req = NULL;

	req = kmalloc(sizeof *req, gfp_flags);
	if (!req)
		return NULL;

	memset(req, 0, sizeof *req);
	req->req.dma = DMA_ADDR_INVALID;
	INIT_LIST_HEAD(&req->queue);

	return &req->req;
}

static void
mcf532x_free_request(struct usb_ep *_ep, struct usb_request *_req)
{

	struct mcf532x_req *req = NULL;

	req = container_of(_req, struct mcf532x_req, req);

	if (_req)
		kfree(req);
}

/*------------------------------------------------------------------ 
 * Allocate an I/O buffer for the ep->req->buf
 * @len: length of the desired buffer 
 * @dma: pointer to the buffer's DMA address; must be valid
 * 	 when gadget layer calls this function, ma is &req->dma
 * @gfp_flags : GFP_* flags to use 
 * Returns a new buffer, or null if one could not be allocated
*---------------------------------------------------------------------*/
static void *
mcf532x_alloc_buffer(struct usb_ep *_ep, unsigned bytes,
		     dma_addr_t * dma, gfp_t gfp_flags)
{
	void *retval = NULL;

	if (!bytes)
		return 0;

	retval = kmalloc(bytes, gfp_flags);
	if (retval)
		*dma = virt_to_phys(retval);
	return retval;
}

/*------------------------------------------------------------------ 
 * Free an I/O buffer for the ep->req->buf
*---------------------------------------------------------------------*/
static void
mcf532x_free_buffer(struct usb_ep *_ep, void *buf,
		    dma_addr_t dma, unsigned bytes)
{
	if (buf)
		kfree(buf);
}

/*-------------------------------------------------------------------------*/

static int
mcf532x_queue_td(struct mcf532x_ep *ep, struct mcf532x_req *req)
{
	int i = ep_index(ep) * 2 + ep_is_in(ep);
	u32 temp, bitmask, tmp_stat;
	struct ep_queue_head *dQH = &ep->udc->ep_qh[i];

	bitmask = (ep_is_in(ep)) ? (1 << (ep_index(ep) + 16)) :
	    (1 << (ep_index(ep)));

	/* check if the pipe is empty */
	if (!(list_empty(&ep->queue))) {
		/* Add td to the end */
		struct mcf532x_req *lastreq;
		lastreq = list_entry(ep->queue.prev, struct mcf532x_req, queue);
		lastreq->tail->next_td_ptr =
		    virt_to_phys(req->head) & DTD_ADDR_MASK;

		/* Read prime bit, if 1 goto done */
		if (MCF_USB0_EPPRIME & bitmask)
			goto out;

		do {
			/* Set ATDTW bit in USBCMD */
			MCF_USB0_USBCMD |= MCF_USB_USBCMD_ATDTW;

			/* Read correct status bit */
			tmp_stat = MCF_USB0_EPSR & bitmask;

		} while (!(MCF_USB0_USBCMD & MCF_USB_USBCMD_ATDTW));

		/* Write ATDTW bit to 0 */
		MCF_USB0_USBCMD &= ~MCF_USB_USBCMD_ATDTW;

		if (tmp_stat)
			goto out;
	}

	/* Write dQH next pointer and terminate bit to 0 */
	temp = virt_to_phys((void *) req->head) &
	    EP_QUEUE_HEAD_NEXT_POINTER_MASK;

	dQH->next_dtd_ptr = temp;

	/* Clear active and halt bit */
	temp = ~(EP_QUEUE_HEAD_STATUS_ACTIVE | EP_QUEUE_HEAD_STATUS_HALT);
	dQH->size_ioc_int_sts &= temp;
	flush_dcache_range((unsigned long) dQH, (unsigned long) dQH +
			   sizeof (struct ep_queue_head));

	/* Prime endpoint by writing 1 to ENDPTPRIME */
	temp = (ep_is_in(ep)) ? (1 << (ep_index(ep) + 16)) :
	    (1 << (ep_index(ep)));

	MCF_USB0_EPPRIME = temp;
      out:
	return 0;
}

static int
mcf532x_build_dtd(struct mcf532x_req *req, unsigned max,
		  struct ep_td_struct **address)
{
	unsigned length;
	u32 swap_temp;
	struct ep_td_struct *dtd;

	/* how big will this packet be? */
	length = min(req->req.length - req->req.actual, max);

	/* Assume CACHELINE alignment garantees 32-byte alignment */
	dtd = kmalloc(sizeof (struct ep_td_struct), GFP_KERNEL);
	/* check alignment */
	if ((u32) dtd & ~DTD_ADDR_MASK)
		panic("Can not allocate aligned memory for dtd");

	memset(dtd, 0, sizeof (struct ep_td_struct));

	/* Fill in the transfer size; set interrupt on every dtd;
	   set active bit */
	dtd->size_ioc_sts = ((length << DTD_LENGTH_BIT_POS) | DTD_IOC
			     | DTD_STATUS_ACTIVE);
	/* Clear reserved field */
	dtd->size_ioc_sts &= ~DTD_RESERVED_FIELDS;

	/* Init all of buffer page pointers */
	swap_temp = (u32) (req->req.buf + req->req.actual);
	dtd->buff_ptr0 = swap_temp;
	dtd->buff_ptr1 = swap_temp + 0x1000;
	dtd->buff_ptr2 = swap_temp + 0x2000;
	dtd->buff_ptr3 = swap_temp + 0x3000;
	dtd->buff_ptr4 = swap_temp + 0x4000;

	req->req.actual += length;
	*address = dtd;
	VDBG("length = %d address= 0x%x", length, (int) dtd);

	return length;
}

static int
mcf532x_req_to_dtd(struct mcf532x_req *req)
{
	unsigned max;
	unsigned count;
	int is_last;
	int is_first = 1;
	struct ep_td_struct *last_addr = NULL, *addr;

	max = EP_MAX_LENGTH_TRANSFER;
	do {
		count = mcf532x_build_dtd(req, max, &addr);

		if (is_first) {
			is_first = 0;
			req->head = addr;
		} else {
			last_addr->next_td_ptr = virt_to_phys(addr);
			flush_dcache_range((unsigned long) last_addr,
					   (unsigned long) last_addr +
					   sizeof (struct ep_td_struct));
			last_addr = addr;
		}
		/* last packet is usually short (or a zlp) */
		if (unlikely(count != max))
			is_last = 1;
		else if (likely(req->req.length != req->req.actual)
			 || req->req.zero)
			is_last = 0;
		else
			is_last = 1;

		req->dtd_count++;
	} while (!is_last);

	addr->next_td_ptr = DTD_NEXT_TERMINATE;
	flush_dcache_range((unsigned long) addr, (unsigned long) addr +
			   sizeof (struct ep_td_struct));

	req->tail = addr;

	return 0;
}

/* queues (submits) an I/O request to an endpoint */
static int
mcf532x_ep_queue(struct usb_ep *_ep, struct usb_request *_req, gfp_t gfp_flags)
{
	struct mcf532x_ep *ep = container_of(_ep, struct mcf532x_ep, ep);
	struct mcf532x_req *req = container_of(_req, struct mcf532x_req, req);
	struct mcf532x_udc *udc;
	unsigned long flags;
	int is_iso = 0;

	/* catch various bogus parameters */
	if (!_req || !req->req.complete || !req->req.buf
	    || !list_empty(&req->queue)) {
		VDBG("%s, bad params\n", __FUNCTION__);
		return -EINVAL;
	}
	if (!_ep || (!ep->desc && ep_index(ep))) {
		VDBG("%s, bad ep\n", __FUNCTION__);
		return -EINVAL;
	}
	if (ep->desc->bmAttributes == USB_ENDPOINT_XFER_ISOC) {
		if (req->req.length > ep->ep.maxpacket)
			return -EMSGSIZE;
		is_iso = 1;
	}

	udc = ep->udc;
	if (!udc->driver || udc->gadget.speed == USB_SPEED_UNKNOWN)
		return -ESHUTDOWN;

	req->ep = ep;

	req->req.status = -EINPROGRESS;
	req->req.actual = 0;
	req->dtd_count = 0;

	spin_lock_irqsave(&udc->lock, flags);

	/* push the dtds to device queue */
	if (!mcf532x_req_to_dtd(req))
		mcf532x_queue_td(ep, req);

	/* EP0 */
	if ((ep_index(ep) == 0))
		udc->ep0_state = DATA_STATE_XMIT;

	/* irq handler advances the queue */
	if (req != NULL)
		list_add_tail(&req->queue, &ep->queue);
	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;
}

/* dequeues (cancels, unlinks) an I/O request from an endpoint */
static int
mcf532x_ep_dequeue(struct usb_ep *_ep, struct usb_request *_req)
{
	struct mcf532x_ep *ep = container_of(_ep, struct mcf532x_ep, ep);
	struct mcf532x_req *req;
	unsigned long flags;

	if (!_ep || !_req)
		return -EINVAL;

	spin_lock_irqsave(&ep->udc->lock, flags);

	/* make sure it's actually queued on this endpoint */
	list_for_each_entry(req, &ep->queue, queue) {
		if (&req->req == _req)
			break;
	}
	if (&req->req != _req) {
		spin_unlock_irqrestore(&ep->udc->lock, flags);
		return -EINVAL;
	}

	done(ep, req, -ECONNRESET);

	spin_unlock_irqrestore(&ep->udc->lock, flags);
	return 0;
}

/*-------------------------------------------------------------------------*/

/*----------------------------------------------------------------- 
 * modify the endpoint halt feature
 * @ep: the non-isochronous endpoint being stalled 
 * @value: 1--set halt  0--clear halt 
 * Returns zero, or a negative error code.
*----------------------------------------------------------------*/
static int
_mcf532x_ep_set_halt(struct usb_ep *_ep, int value)
{

	struct mcf532x_ep *ep = NULL;
	unsigned long flags = 0;
	int status = -EOPNOTSUPP;	/* operation not supported */
	unsigned char ep_dir = 0, ep_num = 0;
	struct mcf532x_udc *udc = NULL;

	ep = container_of(_ep, struct mcf532x_ep, ep);
	udc = ep->udc;
	if (!_ep || !ep->desc) {
		status = -EINVAL;
		goto out;
	}

	if (ep->desc->bmAttributes == USB_ENDPOINT_XFER_ISOC) {
		status = -EOPNOTSUPP;
		goto out;
	}

	/* Attemp to halt IN ep will fail if any transfer requests
	   are still queue */
	if (value && ep_is_in(ep) && !list_empty(&ep->queue)) {

		status = -EAGAIN;
		goto out;
	}

	status = 0;
	ep_dir = ep_is_in(ep) ? USB_SEND : USB_RECV;
	ep_num = (unsigned char) (ep_index(ep));
	spin_lock_irqsave(&ep->udc->lock, flags);
	dr_ep_change_stall(ep_num, ep_dir, value);
	spin_unlock_irqrestore(&ep->udc->lock, flags);

	if (ep_index(ep) == 0) {
		udc->ep0_state = WAIT_FOR_SETUP;
		udc->ep0_dir = 0;
	}
      out:
	VDBG(" %s %s halt stat %d", ep->ep.name, value ?
	     "set" : "clear", status);

	return status;

}
static int
mcf532x_ep_set_halt(struct usb_ep *_ep, int value)
{
	return (_mcf532x_ep_set_halt(_ep, value));
}

static struct usb_ep_ops mcf532x_ep_ops = {
	.enable = mcf532x_ep_enable,
	.disable = mcf532x_ep_disable,

	.alloc_request = mcf532x_alloc_request,
	.free_request = mcf532x_free_request,

	.alloc_buffer = mcf532x_alloc_buffer,
	.free_buffer = mcf532x_free_buffer,

	.queue = mcf532x_ep_queue,
	.dequeue = mcf532x_ep_dequeue,

	.set_halt = mcf532x_ep_set_halt,
};

/*-------------------------------------------------------------------------
		Gadget Driver Layer Operations
-------------------------------------------------------------------------*/

/*************************************************************************
		Gadget Driver Layer Operations
*************************************************************************/

/*----------------------------------------------------------------------
 * Get the current frame number (from DR frame_index Reg )
 *----------------------------------------------------------------------*/
static int
mcf532x_get_frame(struct usb_gadget *gadget)
{
	return (int) (MCF_USB_FRINDEX_FRINDEX(MCF_USB0_FRINDEX));
}

/*-----------------------------------------------------------------------
 * Tries to wake up the host connected to this gadget 
 * 
 * Return : 0-success 
 * Negative-this feature not enabled by host or not supported by device hw
 -----------------------------------------------------------------------*/
static int
mcf532x_wakeup(struct usb_gadget *gadget)
{
	struct mcf532x_udc *udc;
	unsigned long flags;
	int retval = -EHOSTUNREACH;

	udc = container_of(gadget, struct mcf532x_udc, gadget);

	spin_lock_irqsave(&udc->lock, flags);
	if (udc->usb_state == USB_STATE_SUSPENDED) {
		/* NOTE:  OTG spec erratum says that OTG devices may
		 * issue wakeups without host enable.
		 */
		if (udc->wakeup_enabled) {
			DBG("remote wakeup...\n");
			/* Force port resume */
			MCF_USB0_PORTSC |= MCF_USB_PORTSC_FPR;
			retval = 0;
		}

		/* NOTE:  non-OTG systems may use SRP TOO... */
	} else if (udc_controller->usb_state == USB_STATE_NOTATTACHED) {
		if (udc->transceiver)
			retval = otg_start_srp(udc->transceiver);
	}
	spin_unlock_irqrestore(&udc->lock, flags);

	return retval;
}

/* sets the device selfpowered feature
 * this affects the device status reported by the hw driver
 * to reflect that it now has a local power supply
 * usually device hw has register for this feature 
 */
static int
mcf532x_set_selfpowered(struct usb_gadget *gadget, int is_selfpowered)
{
	struct mcf532x_udc *udc;
	unsigned long flags;

	udc = container_of(gadget, struct mcf532x_udc, gadget);
	spin_lock_irqsave(&udc->lock, flags);
	udc->selfpowered = (is_selfpowered != 0);
	spin_unlock_irqrestore(&udc->lock, flags);

	return 0;;
}

/* Notify controller that VBUS is powered, Called by whatever 
   detects VBUS sessions */
static int
mcf532x_vbus_session(struct usb_gadget *gadget, int is_active)
{
	struct mcf532x_udc *udc;
	unsigned long flags;

	udc = container_of(gadget, struct mcf532x_udc, gadget);
	spin_lock_irqsave(&udc->lock, flags);
	VDBG("VBUS %s\n", is_active ? "on" : "off");
	udc->vbus_active = (is_active != 0);

	if (can_pullup(udc)) {
		pullup_enable(udc);
		udc_controller->usb_state = USB_STATE_ATTACHED;
	} else {
		pullup_disable(udc);
		udc_controller->usb_state = USB_STATE_NOTATTACHED;
	}
	udc_controller->ep0_state = WAIT_FOR_SETUP;
	udc_controller->ep0_dir = 0;

	spin_unlock_irqrestore(&udc->lock, flags);
	return 0;
}

/* constrain controller's VBUS power usage 
 * This call is used by gadget drivers during SET_CONFIGURATION calls,
 * reporting how much power the device may consume.  For example, this
 * could affect how quickly batteries are recharged.
 *
 * Returns zero on success, else negative errno.
 */
static int
mcf532x_vbus_draw(struct usb_gadget *gadget, unsigned mA)
{
	struct mcf532x_udc *udc;

	udc = container_of(gadget, struct mcf532x_udc, gadget);
	if (udc->transceiver)
		return otg_set_power(udc->transceiver, mA);
	return -ENOTSUPP;
}

/* Change Data+ pullup status
 * this func is used by usb_gadget_connect/disconnet 
 */
static int
mcf532x_pullup(struct usb_gadget *gadget, int is_on)
{
	return -ENOTSUPP;
}

/* defined in usb_gadget.h */
static struct usb_gadget_ops mcf532x_gadget_ops = {
	.get_frame = mcf532x_get_frame,
	.wakeup = mcf532x_wakeup,
	.set_selfpowered = mcf532x_set_selfpowered,
	.vbus_session = mcf532x_vbus_session,
	.vbus_draw = mcf532x_vbus_draw,
	.pullup = mcf532x_pullup,
};

static void
Ep0Stall(struct mcf532x_udc *udc)
{
	MCF_USB0_EPCR(0) |= MCF_USB_EPCR_TXS | MCF_USB_EPCR_RXS;
	udc->ep0_state = WAIT_FOR_SETUP;
	udc->ep0_dir = 0;
}

/* if direction is EP_IN, the status is Device->Host
 * if direction is EP_OUT, the status transaction is Device<-Host
 */
static int
ep0_prime_status(struct mcf532x_udc *udc, int direction)
{
	struct mcf532x_req *req = udc->status_req;
	struct mcf532x_ep *ep;
	int status = 0;

	if (direction == EP_DIR_IN)
		udc->ep0_dir = USB_DIR_IN;
	else
		udc->ep0_dir = USB_DIR_OUT;

	ep = &udc->eps[0];
	udc->ep0_state = WAIT_FOR_OUT_STATUS;

	req->ep = ep;
	req->req.length = 0;
	req->req.status = -EINPROGRESS;
	req->req.actual = 0;
	req->req.complete = NULL;

	if ((mcf532x_req_to_dtd(req) == 0))
		status = mcf532x_queue_td(ep, req);
	if (status)
		DBG("Can't get control status request \n");
	list_add_tail(&req->queue, &ep->queue);

	return status;
}

static int
udc_reset_ep_queue(struct mcf532x_udc *udc, u8 pipe)
{

	struct mcf532x_ep *ep = get_ep_by_pipe(udc, pipe);

	/* FIXME: collect completed requests? */
	if (!ep->name)
		return 0;

	nuke(ep, -ECONNRESET);

	return 0;
}

static void
setup_received_irq(struct mcf532x_udc *udc, struct usb_ctrlrequest *setup)
{
#define	w_value le16_to_cpu(setup->wValue)
#define	w_index le16_to_cpu(setup->wIndex)
#define	w_length le16_to_cpu(setup->wLength)

	udc_reset_ep_queue(udc, 0);

	/* We asume setup only occurs on EP0 */
	if ((setup->bRequestType & USB_DIR_IN) || (w_length == 0))
		udc->ep0_dir = USB_DIR_IN;
	else
		udc->ep0_dir = USB_DIR_OUT;

	if ((setup->bRequestType & USB_TYPE_MASK) == USB_TYPE_STANDARD) {
		switch (setup->bRequest) {

		case USB_REQ_GET_STATUS:{
				u16 usb_status = 0;
				struct mcf532x_req *req;
				struct mcf532x_ep *ep;
				u8 ep_number;
				u8 ep_index;
				int status = 0;

				if ((w_length != 2) || (w_value != 0)) {
					Ep0Stall(udc);
					return;
				}

				ep = &udc->eps[0];

				req =
				    container_of(mcf532x_alloc_request
						 (&ep->ep, GFP_KERNEL),
						 struct mcf532x_req, req);
				req->req.length = 2;
				req->req.status = -EINPROGRESS;
				req->req.actual = 0;

				switch (setup->bRequestType & USB_RECIP_MASK) {
				case USB_RECIP_DEVICE:
					usb_status = udc->selfpowered;	/* Report selfpoered status. */
					usb_status |=
					    (u16) udc->wakeup_enabled << 1;
					break;
				case USB_RECIP_INTERFACE:
					usb_status = 0;
					break;
				case USB_RECIP_ENDPOINT:
					ep_number =
					    w_index & USB_ENDPOINT_NUMBER_MASK;
					ep_index =
					    ep_number * 2 +
					    (((w_index & USB_DIR_IN) ==
					      USB_DIR_IN) ? 1 : 0);

					if ((ep_number >
					     (USB_MAX_ENDPOINTS - 1))
					    || ((ep_number != 0)
						&& (udc->eps[ep_index].desc ==
						    0))
					    ) {
						Ep0Stall(udc);
						return;
					};

					if ((w_index & USB_DIR_IN) ==
					    USB_DIR_IN)
						usb_status =
						    (MCF_USB0_EPCR(ep_number) |
						     MCF_USB_EPCR_TXS) ? 1 : 0;
					else
						usb_status =
						    (MCF_USB0_EPCR(ep_number) |
						     MCF_USB_EPCR_RXS) ? 1 : 0;

					break;
				default:
					Ep0Stall(udc);
					return;
				}

				req->req.buf = &usb_status;

				/* data phase */
				if ((mcf532x_req_to_dtd(req) == 0))
					status = mcf532x_queue_td(ep, req);
				if (status) {
					DBG("Can't respond to getstatus request \n");
					Ep0Stall(udc);
				} else
					udc->ep0_state = DATA_STATE_XMIT;
				return;
			}
		case USB_REQ_SET_ADDRESS:
			if (((setup->bRequestType & USB_RECIP_MASK) !=
			     USB_RECIP_DEVICE) || (w_length != 0)
			    || (w_index != 0)) {
				Ep0Stall(udc);
				return;
			}

			/* Save the new address to device struct */
			udc->device_address = (u8) w_value;

			/* Update usb state */
			udc->usb_state = USB_STATE_ADDRESS;

			/* Status phase */
			if (ep0_prime_status(udc, EP_DIR_IN))
				Ep0Stall(udc);
			udc->ep0_state = WAIT_FOR_SETUP;
			return;

		case USB_REQ_SET_CONFIGURATION:
			if (((setup->bRequestType & USB_RECIP_MASK) !=
			     USB_RECIP_DEVICE) || (w_length != 0)
			    || (w_index != 0)) {
				Ep0Stall(udc);
				return;
			}

			if (udc->driver->
			    setup(&udc->gadget, &udc->local_setup_buff) >= 0) {
				/* gadget layer deal with the status phase */
				udc->usb_state = USB_STATE_CONFIGURED;
				udc->ep0_state = WAIT_FOR_OUT_STATUS;
			} else {
				Ep0Stall(udc);
				return;
			}

			return;
			/* Requests with no data phase */
		case USB_REQ_CLEAR_FEATURE:
		case USB_REQ_SET_FEATURE:{
				int status = 0;

				if (w_length != 0) {
					Ep0Stall(udc);
					return;
				}

				switch (setup->bRequestType & USB_RECIP_MASK) {
				case USB_RECIP_DEVICE:
					switch (w_value) {
					case 1:
						if (setup->bRequest ==
						    USB_REQ_SET_FEATURE) {
							udc->wakeup_enabled = 1;
						} else {
							udc->wakeup_enabled = 0;
						}
						break;
					case USB_DEVICE_TEST_MODE:	/* TBD TEST_MODE */
						break;
					case USB_DEVICE_B_HNP_ENABLE:
						if (udc->gadget.is_otg &&
						    (setup->bRequest ==
						     USB_REQ_SET_FEATURE)) {
							udc->gadget.
							    b_hnp_enable = 1;
						} else {
							Ep0Stall(udc);
							return;
						}
						break;
					case USB_DEVICE_A_HNP_SUPPORT:
						if (udc->gadget.is_otg &&
						    (setup->bRequest ==
						     USB_REQ_SET_FEATURE)) {
							udc->gadget.
							    a_hnp_support = 1;
						} else {
							Ep0Stall(udc);
							return;
						}
						break;
					case USB_DEVICE_A_ALT_HNP_SUPPORT:
						if (udc->gadget.is_otg &&
						    (setup->bRequest ==
						     USB_REQ_SET_FEATURE)) {
							udc->gadget.
							    a_alt_hnp_support =
							    1;
						} else {
							Ep0Stall(udc);
							return;
						}
						break;
					default:
						Ep0Stall(udc);
						return;
					}
					break;
				case USB_RECIP_INTERFACE:
					break;
				case USB_RECIP_ENDPOINT:{
						struct mcf532x_ep *ep;
						int ep_number;
						int ep_index;

						ep_number =
						    w_index &
						    USB_ENDPOINT_NUMBER_MASK;
						ep_index =
						    ep_number * 2 +
						    (((w_index & USB_DIR_IN) ==
						      USB_DIR_IN) ? 1 : 0);

						if ((w_value != 0)
						    || (ep_number >
							(USB_MAX_ENDPOINTS -
							 1))) {
							Ep0Stall(udc);
							return;
						};

						ep = &udc->eps[ep_index];

						if (setup->bRequest ==
						    USB_REQ_SET_FEATURE) {
							status =
							    _mcf532x_ep_set_halt
							    (&ep->ep, 1);
						} else {
							status =
							    _mcf532x_ep_set_halt
							    (&ep->ep, 0);
						}
						break;
					}
				default:
					Ep0Stall(udc);
					return;
				}
				if (status == 0) {
					/* send status only if _mcf532x_ep_set_halt success */
					if (ep0_prime_status(udc, EP_DIR_IN))
						Ep0Stall(udc);
				}

				return;
			}

		}		/*switch (setup->bRequest) */
	}

	if (udc->driver->setup(&udc->gadget, &udc->local_setup_buff) != 0) {
		Ep0Stall(udc);
	} else {
		if (setup->bRequestType & USB_DIR_IN)
			udc->ep0_state = DATA_STATE_XMIT;
		else
			udc->ep0_state = DATA_STATE_RECV;
	}
}

static void
ep0_req_complete(struct mcf532x_udc *udc, struct mcf532x_ep *ep0,
		 struct mcf532x_req *req)
{
	if (udc->usb_state == USB_STATE_ADDRESS) {
		/* Set the new address */
		u32 new_address = (u32) udc->device_address;
		MCF_USB0_DEVICEADDR = MCF_USB_DEVICEADDR_USBADR(new_address);
	}

	switch (udc->ep0_state) {
	case DATA_STATE_XMIT:
		done(ep0, req, 0);
		/* receive status phase */
		if (ep0_prime_status(udc, EP_DIR_OUT))
			Ep0Stall(udc);
		break;
	case DATA_STATE_RECV:
		done(ep0, req, 0);
		/* send status phase */
		if (ep0_prime_status(udc, EP_DIR_IN))
			Ep0Stall(udc);
		break;
	case WAIT_FOR_OUT_STATUS:
		done(ep0, req, 0);
		udc->ep0_state = WAIT_FOR_SETUP;
		break;
	case WAIT_FOR_SETUP:
		VDBG("Unexpected interrupt");
		break;
	default:
		Ep0Stall(udc);
		break;
	}
}

static void
tripwire_handler(struct mcf532x_udc *udc, u8 ep_num, u32 * buffer_ptr)
{
	struct ep_queue_head *qh;

	qh = &udc->ep_qh[ep_num * 2 + EP_DIR_OUT];

	/* Clear bit in ENDPTSETUPSTAT */
	MCF_USB0_EPSETUPSR |= MCF_USB_EPSETUPSR_EPSETUPSTAT(1 << ep_num);
	/* while a hazard exists when setup package arrives */
	do {
		/* Set Setup Tripwire */
		MCF_USB0_USBCMD |= MCF_USB_USBCMD_SUTW;

		/* Copy the setup packet to local buffer */
		buffer_ptr[0] = le32_to_cpu(qh->setup_buffer[0]);
		buffer_ptr[1] = le32_to_cpu(qh->setup_buffer[1]);
	} while (!(MCF_USB0_USBCMD & MCF_USB_USBCMD_SUTW));

	/* Clear Setup Tripwire */
	MCF_USB0_USBCMD &= ~MCF_USB_USBCMD_SUTW;

}

/*process-ep_req(): free the completed Tds for this req */
static int
process_ep_req(struct mcf532x_udc *udc, int pipe, struct mcf532x_req *curr_req)
{
	struct ep_td_struct *curr_td, *tmp_td;
	int td_complete, actual, remaining_length, j, tmp;
	int status = 0;
	int errors = 0;
	struct ep_queue_head *curr_qh = &udc->ep_qh[pipe];
	int direction = pipe % 2;

	curr_td = curr_req->head;
	td_complete = 0;
	actual = curr_req->req.length;

	for (j = 0; j < curr_req->dtd_count; j++) {
		remaining_length = ((curr_td->size_ioc_sts
				     & DTD_PACKET_SIZE) >> DTD_LENGTH_BIT_POS);
		actual -= remaining_length;

		if ((errors = curr_td->size_ioc_sts & DTD_ERROR_MASK)) {
			if (errors & DTD_STATUS_HALTED) {
				DBG("dTD error %08x \n", errors);
				/* Clear the errors and Halt condition */
				tmp = curr_qh->size_ioc_int_sts;
				tmp &= ~errors;
				curr_qh->size_ioc_int_sts = tmp;
				status = -EPIPE;
				/*FIXME clearing active bit, update 
				 * nextTD pointer re-prime ep */
				break;
			}
			if (errors & DTD_STATUS_DATA_BUFF_ERR) {
				VDBG("Transfer overflow");
				status = -EPROTO;
				break;
			} else if (errors & DTD_STATUS_TRANSACTION_ERR) {
				VDBG("ISO error");
				status = -EILSEQ;
				break;
			} else
				ERR("Unknown error has occured (0x%x)!\r\n",
				    errors);

		} else if ((curr_td->size_ioc_sts) & DTD_STATUS_ACTIVE) {
			VDBG("Request not wholly complete");
			status = REQ_UNCOMPLETE;
			return status;
		} else if (remaining_length)
			if (direction) {
				VDBG("Transmit dTD remaining length not zero");
				status = -EPROTO;
				break;
			} else {
				td_complete += 1;
				break;
		} else {
			td_complete += 1;
			VDBG("dTD transmitted successful ");
		}

		if (j != curr_req->dtd_count - 1)
			curr_td =
			    (struct ep_td_struct *) phys_to_virt(curr_td->
								 next_td_ptr &
								 DTD_ADDR_MASK);

	}

	if (status)
		return status;

	curr_req->req.actual = actual;

	/* Free dtd for completed/error request */
	curr_td = curr_req->head;
	for (j = 0; j < curr_req->dtd_count; j++) {
		tmp_td = curr_td;
		if (j != curr_req->dtd_count - 1)
			curr_td =
			    (struct ep_td_struct *) phys_to_virt(curr_td->
								 next_td_ptr &
								 DTD_ADDR_MASK);
		kfree(tmp_td);
	}

	return status;

}

static void
dtd_complete_irq(struct mcf532x_udc *udc)
{
	u32 bit_pos;
	int i, ep_num, direction, bit_mask, status;
	struct mcf532x_ep *curr_ep;
	struct mcf532x_req *curr_req, *temp_req;

	/* Clear the bits in the register */
	bit_pos = MCF_USB0_EPCOMPLETE;
	MCF_USB0_EPCOMPLETE = bit_pos;

	if (!bit_pos)
		return;

	for (i = 0; i < USB_MAX_ENDPOINTS * 2; i++) {
		ep_num = i >> 1;
		direction = i % 2;

		bit_mask = 1 << (ep_num + 16 * direction);

		if (!(bit_pos & bit_mask))
			continue;

		curr_ep = get_ep_by_pipe(udc, i);

		/* If the ep is configured */
		if (curr_ep->name == NULL) {
			WARN("Invalid EP?");
			continue;
		}

		/* search all mcf532x_reqs of ep */
		list_for_each_entry_safe(curr_req, temp_req, &curr_ep->queue,
					 queue) {
			status = process_ep_req(udc, i, curr_req);
			if (status == REQ_UNCOMPLETE) {
				VDBG("Not all tds are completed in the req");
				break;
			}

			if (ep_num == 0) {
				ep0_req_complete(udc, curr_ep, curr_req);
				break;
			} else
				done(curr_ep, curr_req, status);

		}

	}
}

static void
port_change_irq(struct mcf532x_udc *udc)
{
	u32 speed;

	if (udc->bus_reset)
		udc->bus_reset = 0;

	/* Bus resetting is finished */
	if (!(MCF_USB0_PORTSC & MCF_USB_PORTSC_PR)) {
		/* Get the speed */
		speed = MCF_USB0_PORTSC & MCF_USB_PORTSC_PSPD;
		switch (speed) {
		case MCF_USB_PORTSC_PSPD_HIGH:
			udc->gadget.speed = USB_SPEED_HIGH;
			break;
		case MCF_USB_PORTSC_PSPD_FULL:
			udc->gadget.speed = USB_SPEED_FULL;
			break;
		case MCF_USB_PORTSC_PSPD_LOW:
			udc->gadget.speed = USB_SPEED_LOW;
			break;
		default:
			udc->gadget.speed = USB_SPEED_UNKNOWN;
			break;
		}
	}

	/* Update USB state */
	if (!udc->resume_state)
		udc->usb_state = USB_STATE_DEFAULT;
}

static void
suspend_irq(struct mcf532x_udc *udc)
{
	udc->resume_state = udc->usb_state;
	udc->usb_state = USB_STATE_SUSPENDED;

	/* report suspend to the driver ,serial.c not support this */
	if (udc->gadget.speed != USB_SPEED_UNKNOWN && udc->driver->suspend) {
		udc->driver->suspend(&udc->gadget);
	}
}

static void
resume_irq(struct mcf532x_udc *udc)
{
	udc->usb_state = udc->resume_state;
	udc->resume_state = 0;

	/* report resume to the driver , serial.c not support this */
	if (udc->driver->resume)
		udc->driver->resume(&udc->gadget);

}

static int
reset_queues(struct mcf532x_udc *udc)
{
	u8 pipe;

	for (pipe = 0; pipe < udc->max_pipes; pipe++)
		udc_reset_ep_queue(udc, pipe);

	/* report disconnect; the driver is already quiesced */
	if ((udc->gadget.speed != USB_SPEED_UNKNOWN) && udc->driver->disconnect)
		udc->driver->disconnect(&udc->gadget);

	return 0;
}

/*
 *  Interrupt handler for USB reset received
 */

static void
reset_irq(struct mcf532x_udc *udc)
{
	/* Clear the device address */
	MCF_USB0_DEVICEADDR = 0;
	udc->device_address = 0;

	/* Clear usb state */
	udc->usb_state = USB_STATE_DEFAULT;

	udc->wakeup_enabled = 0;

	/* Clear all the setup token semaphores */
	MCF_USB0_EPSETUPSR = MCF_USB0_EPSETUPSR;
	/* Clear all the endpoint complete status bits */
	MCF_USB0_EPCOMPLETE = MCF_USB0_EPCOMPLETE;

	/* Write 1s to the Flush register */
	MCF_USB0_EPFLUSH = 0xFFFFFFFF;

	if (MCF_USB0_PORTSC & MCF_USB_PORTSC_PR) {
		/* Bus is reseting */
		udc->bus_reset = 1;
		udc->ep0_state = WAIT_FOR_SETUP;
		udc->ep0_dir = 0;
		/* Reset all the queues, include XD, dTD, EP queue
		 * head and TR Queue */
		reset_queues(udc);
	} else {
		/* initialize usb hw reg except for regs for EP, not
		 * touch usbintr reg */
		dr_controller_setup(udc->ep_qh);

		/* Reset all internal used Queues */
		reset_queues(udc);

		ep0_dr_and_qh_setup(udc);

		/* Enable DR IRQ reg, Set Run bit, change udc state */
		dr_controller_run(udc_controller);
		udc->usb_state = USB_STATE_ATTACHED;
		udc->ep0_state = WAIT_FOR_SETUP;
		udc->ep0_dir = 0;
	}

	udc->gadget.speed = USB_SPEED_UNKNOWN;
	udc->gadget.b_hnp_enable = 0;
	udc->gadget.a_hnp_support = 0;
	udc->gadget.a_alt_hnp_support = 0;
}

/*
 * USB device controller interrupt handler
 */
static irqreturn_t
mcf532x_udc_irq(int irq, void *_udc, struct pt_regs *r)
{
	struct mcf532x_udc *udc = _udc;
	u32 irq_src;
	irqreturn_t status = IRQ_NONE;
	unsigned long flags;

	spin_lock_irqsave(&udc->lock, flags);

	if ((MCF_USB0_USBMODE & MCF_USB_USBMODE_CM(0xF)) ==
	    MCF_USB_USBMODE_CM_DEVICE) {
		flush_cache_all();	/* Cache issue workaround */
		irq_src = MCF_USB0_USBSTS & MCF_USB0_USBINTR;

		/* Clear notification bits */
		MCF_USB0_USBSTS &= irq_src;

		/* SOF (for ISO transfer) */
		if (irq_src & MCF_USB_USBSTS_SRI) {
			status = IRQ_HANDLED;
		}

		/* Port Change */
		if (irq_src & MCF_USB_USBSTS_PCI) {
			port_change_irq(udc);
			status = IRQ_HANDLED;
		}

		/* Reset Received */
		if (irq_src & MCF_USB_USBSTS_URI) {
			reset_irq(udc);
			status = IRQ_HANDLED;
		}

		/* Sleep Enable (Suspend) */
		if (irq_src & MCF_USB_USBSTS_SLI) {
			suspend_irq(udc);
			status = IRQ_HANDLED;
		} else if (udc->resume_state) {
			resume_irq(udc);
		}

		/* USB Interrupt */
		if (irq_src & MCF_USB_USBSTS_UI) {
			/* Setup package, we only support ep0 as control ep */
			if (MCF_USB0_EPSETUPSR &
			    MCF_USB_EPSETUPSR_EPSETUPSTAT(1)) {

				tripwire_handler(udc, 0,
						 (u32 *) (&udc->
							  local_setup_buff));
				setup_received_irq(udc, &udc->local_setup_buff);
				status = IRQ_HANDLED;
			}

			/* completion of dtd */
			if (MCF_USB0_EPCOMPLETE) {
				dtd_complete_irq(udc);
				status = IRQ_HANDLED;
			}
		}

		if (irq_src & (MCF_USB_USBSTS_UEI | MCF_USB_USBSTS_SEI)) {
			VDBG("Error IRQ %x ", irq_src);
		}
	}
	spin_unlock_irqrestore(&udc->lock, flags);

	return status;
}

/*----------------------------------------------------------------*
 * tell the controller driver about gadget layer driver
 * The driver's bind function will be called to bind it to a gadget.
 * @driver: for example fsg_driver from file_storage.c
*----------------------------------------------------------------*/
int
usb_gadget_register_driver(struct usb_gadget_driver *driver)
{
	int retval = -ENODEV;
	unsigned long flags = 0;

	/* standard operations */
	if (!udc_controller)
		return -ENODEV;

	if (!driver || ((driver->speed != USB_SPEED_FULL)
			&& (driver->speed != USB_SPEED_HIGH))
	    || !driver->bind || !driver->disconnect || !driver->setup)
		return -EINVAL;

	if (udc_controller->driver)
		return -EBUSY;

	/* lock is needed but whether should use this lock or another */
	spin_lock_irqsave(&udc_controller->lock, flags);

	driver->driver.bus = 0;
	udc_controller->softconnect = 1;
	/* hook up the driver */
	udc_controller->driver = driver;
	udc_controller->gadget.dev.driver = &driver->driver;
	spin_unlock_irqrestore(&udc_controller->lock, flags);

	retval = driver->bind(&udc_controller->gadget);
	if (retval) {
		VDBG("bind to %s --> %d", driver->driver.name, retval);
		udc_controller->gadget.dev.driver = 0;
		udc_controller->driver = 0;
		goto out;
	}

	/* Enable DR IRQ reg and Set usbcmd reg  Run bit */
	if(udc_controller->transceiver)
	{
		mcf532x_udc_suspend(&udc_controller->gadget.dev, PMSG_SUSPEND);

		/* Export udc suspend/resume call to OTG */
		udc_controller->gadget.dev.driver->suspend = mcf532x_udc_suspend;
		udc_controller->gadget.dev.driver->resume = mcf532x_udc_resume;

		retval = otg_set_peripheral(udc_controller->transceiver,
					    &udc_controller->gadget);
		if (retval < 0) {
			ERR("can't bind to transceiver\n");
			driver->unbind(&udc_controller->gadget);
			udc_controller->gadget.dev.driver = NULL;
			udc_controller->driver = NULL;
			goto out;
		}
	}
	else{
		dr_controller_run(udc_controller);
		udc_controller->usb_state = USB_STATE_ATTACHED;
		udc_controller->ep0_state = WAIT_FOR_SETUP;
		udc_controller->ep0_dir = 0;
	}

	INFO("mcf532x_udc: %s bind to driver %s \n",
	     udc_controller->gadget.name, driver->driver.name);

      out:return retval;

}

EXPORT_SYMBOL(usb_gadget_register_driver);

int
usb_gadget_unregister_driver(struct usb_gadget_driver *driver)
{
	struct mcf532x_ep *loop_ep;
	unsigned long flags;

	if (!udc_controller)
		return -ENODEV;

	if (!driver || driver != udc_controller->driver)
		return -EINVAL;

	if (udc_controller->transceiver)
		(void) otg_set_peripheral(udc_controller->transceiver, NULL);

	/* stop DR, disable intr */
	dr_controller_stop(udc_controller);

	/* stand operation */
	spin_lock_irqsave(&udc_controller->lock, flags);
	udc_controller->gadget.speed = USB_SPEED_UNKNOWN;
	nuke(&udc_controller->eps[0], -ESHUTDOWN);
	list_for_each_entry(loop_ep, &udc_controller->gadget.ep_list,
			    ep.ep_list)
	    nuke(loop_ep, -ESHUTDOWN);
	spin_unlock_irqrestore(&udc_controller->lock, flags);

	/* unbind gadget and unhook driver. */
	driver->unbind(&udc_controller->gadget);
	udc_controller->gadget.dev.driver = 0;
	udc_controller->driver = 0;

	INFO("unregistered gadget driver '%s'\r\n", driver->driver.name);
	return 0;
}

EXPORT_SYMBOL(usb_gadget_unregister_driver);

/*-------------------------------------------------------------------------
		PROC File System Support
-------------------------------------------------------------------------*/
#ifdef CONFIG_USB_GADGET_DEBUG_FILES

#include <linux/seq_file.h>

static const char proc_filename[] = "driver/mcf532x_udc";

static int
mcf532x_proc_read(char *page, char **start, off_t off, int count,
		  int *eof, void *_dev)
{
	char *buf = page;
	char *next = buf;
	unsigned size = count;
	unsigned long flags;
	int t, i;
	u32 tmp_reg;
	struct mcf532x_ep *ep = NULL;
	struct mcf532x_req *req;

	struct mcf532x_udc *udc = udc_controller;
	if (off != 0)
		return 0;

	spin_lock_irqsave(&udc->lock, flags);

	/* ------basic driver infomation ---- */
	t = scnprintf(next, size,
		      DRIVER_DESC "\n" "%s version: %s\n"
		      "Gadget driver: %s\n\n", driver_name, DRIVER_VERSION,
		      udc->driver ? udc->driver->driver.name : "(none)");
	size -= t;
	next += t;

	/* ------ DR Registers ----- */
	tmp_reg = MCF_USB0_USBCMD;
	t = scnprintf(next, size,
		      "USBCMD reg:\n" "SetupTW: %d\n" "Run/Stop: %s\n\n",
		      (tmp_reg & MCF_USB_USBCMD_SUTW) ? 1 : 0,
		      (tmp_reg & MCF_USB_USBCMD_RS) ? "Run" : "Stop");
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_USBSTS;
	t = scnprintf(next, size,
		      "USB Status Reg:\n" "Dr Suspend: %d\n"
		      "Reset Received: %d" "System Error: %s\n"
		      "USB Error Interrupt: %s\n\n",
		      (tmp_reg & MCF_USB_USBSTS_SLI) ? 1 : 0,
		      (tmp_reg & MCF_USB_USBSTS_URI) ? 1 : 0,
		      (tmp_reg & MCF_USB_USBSTS_SEI) ? "Err" : "Normal",
		      (tmp_reg & MCF_USB_USBSTS_UEI) ? "Err detected" :
		      "No err");
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_USBINTR;
	t = scnprintf(next, size,
		      "USB Intrrupt Enable Reg:\n"
		      "Sleep Enable: %d\n" "SOF Received Enable: %d\n"
		      "Reset Enable: %d\n" "System Error Enable: %d\n"
		      "Port Change Dectected Enable: %d\n"
		      "USB Error Intr Enable: %d\n"
		      "USB Intr Enable: %d\n\n",
		      (tmp_reg & MCF_USB_USBINTR_SLE) ? 1 : 0,
		      (tmp_reg & MCF_USB_USBINTR_SRE) ? 1 : 0,
		      (tmp_reg & MCF_USB_USBINTR_URE) ? 1 : 0,
		      (tmp_reg & MCF_USB_USBINTR_SEE) ? 1 : 0,
		      (tmp_reg & MCF_USB_USBINTR_PCE) ? 1 : 0,
		      (tmp_reg & MCF_USB_USBINTR_UEE) ? 1 : 0,
		      (tmp_reg & MCF_USB_USBINTR_UE) ? 1 : 0);
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_FRINDEX;
	t = scnprintf(next, size,
		      "USB Frame Index Reg:" "Frame Number is 0x%x\n\n",
		      (MCF_USB_FRINDEX_FRINDEX(tmp_reg)));
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_DEVICEADDR;
	t = scnprintf(next, size,
		      "USB Device Address Reg:" "Device Addr is 0x%x\n\n",
		      (tmp_reg & MCF_USB_DEVICEADDR_USBADR(0x7F)));
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_EPLISTADDR;
	t = scnprintf(next, size,
		      "USB Endpoint List Address Reg:"
		      "Device Addr is 0x%x\n\n",
		      (tmp_reg & MCF_USB_EPLISTADDR_EPBASE(0xFFFFFFFF)));
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_PORTSC;
	t = scnprintf(next, size,
		      "USB Port Status&Control Reg:\n"
		      "Port Transceiver Type : %s\n" "Port Speed: %s \n"
		      "PHY Low Power Suspend: %s\n" "Port Reset: %s\n"
		      "Port Suspend Mode: %s \n" "Over-current Change: %s\n"
		      "Port Enable/Disable Change: %s\n"
		      "Port Enabled/Disabled: %s\n"
		      "Current Connect Status: %s\n\n", ( {
							 char *s;
							 switch (tmp_reg &
								 MCF_USB_PORTSC_PTS
								 (3)) {
case MCF_USB_PORTSC_PTS_ULPI:
s = "ULPI "; break; case MCF_USB_PORTSC_PTS_FS_LS:
s = "FS/LS Serial"; break; default:
							 s = "None"; break;}
							 s;}
		      ), ( {
			  char *s; switch (tmp_reg & MCF_USB_PORTSC_PSPD) {
case MCF_USB_PORTSC_PSPD_FULL:
s = "Full Speed"; break; case MCF_USB_PORTSC_PSPD_LOW:
s = "Low Speed"; break; case MCF_USB_PORTSC_PSPD_HIGH:
s = "High Speed"; break; default:
			  s = "Undefined"; break;}
			  s;}
		      ),
		      (tmp_reg & MCF_USB_PORTSC_PHCD) ?
		      "Normal PHY mode" : "Low power mode",
		      (tmp_reg & MCF_USB_PORTSC_PR) ? "In Reset" :
		      "Not in Reset",
		      (tmp_reg & MCF_USB_PORTSC_SUSP) ? "In " : "Not in",
		      (tmp_reg & MCF_USB_PORTSC_OCC) ? "Dected" :
		      "No",
		      (tmp_reg & MCF_USB_PORTSC_OCC) ? "Disable" :
		      "Not change",
		      (tmp_reg & MCF_USB_PORTSC_PE) ? "Enable" :
		      "Not correct",
		      (tmp_reg & MCF_USB_PORTSC_CCS) ? "Attached" : "Not-Att") ;
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_USBMODE;
	t = scnprintf(next, size, "USB Mode Reg:" "Controller Mode is : %s\n\n", ( {
										  char
										  *s;
										  switch
										  (tmp_reg
										   &
										   MCF_USB_USBMODE_CM_HOST)
										  {
case MCF_USB_USBMODE_CM_IDLE:
s = "Idle"; break; case MCF_USB_USBMODE_CM_DEVICE:
s = "Device Controller"; break; case MCF_USB_USBMODE_CM_HOST:
s = "Host Controller"; break; default:
										  s
										  =
										  "None";
										  break;}
										  s;}
		      )) ;
	size -= t;
	next += t;

	tmp_reg = MCF_USB0_EPSETUPSR;
	t = scnprintf(next, size,
		      "Endpoint Setup Status Reg:" "SETUP on ep 0x%x\n\n",
		      (tmp_reg & MCF_USB_EPSETUPSR_EPSETUPSTAT(0xFFFFFFFF)));
	size -= t;
	next += t;

	for (i = 0; i < USB_MAX_ENDPOINTS; i++) {
		tmp_reg = MCF_USB0_EPCR(i);
		t = scnprintf(next, size, "EP Ctrl Reg [0x%x]: = [0x%x]\n",
			      i, tmp_reg);
		size -= t;
		next += t;
	}

	tmp_reg = MCF_USB0_EPPRIME;
	t = scnprintf(next, size, "EP Prime Reg = [0x%x]\n", tmp_reg);
	size -= t;
	next += t;

	/* mcf532x_udc, mcf532x_ep, mcf532x_request structure information */
	ep = &udc->eps[0];
	t = scnprintf(next, size, "For %s Maxpkt is 0x%x index is 0x%x\n",
		      ep->ep.name, ep_maxpacket(ep), ep_index(ep));
	size -= t;
	next += t;

	if (list_empty(&ep->queue)) {
		t = scnprintf(next, size, "its req queue is empty\n\n");
		size -= t;
		next += t;
	} else {
		list_for_each_entry(req, &ep->queue, queue) {
			t = scnprintf(next, size,
				      "req %p actual 0x%x length 0x%x  buf %p\n",
				      &req->req, req->req.actual,
				      req->req.length, req->req.buf);
			size -= t;
			next += t;
		}
	}
	/* other gadget->eplist ep */
	list_for_each_entry(ep, &udc->gadget.ep_list, ep.ep_list) {
		if (ep->desc) {
			t = scnprintf(next, size,
				      "\nFor %s Maxpkt is 0x%x index is 0x%x\n",
				      ep->ep.name,
				      ep_maxpacket(ep), ep_index(ep));
			size -= t;
			next += t;

			if (list_empty(&ep->queue)) {
				t = scnprintf(next, size,
					      "its req queue is empty\n\n");
				size -= t;
				next += t;
			} else {
				list_for_each_entry(req, &ep->queue, queue) {
					t = scnprintf(next, size,
						      "req %p actual 0x%x length"
						      "0x%x  buf %p\n",
						      &req->req,
						      req->req.actual,
						      req->req.length,
						      req->req.buf);
					size -= t;
					next += t;
				}
			}
		}
	}

	spin_unlock_irqrestore(&udc->lock, flags);

	*eof = 1;
	return count - size;
}

#define create_proc_file()	create_proc_read_entry(proc_filename, \
				0, NULL, mcf532x_proc_read, NULL)

#define remove_proc_file()	remove_proc_entry(proc_filename, NULL)

#else				/* !CONFIG_USB_GADGET_DEBUG_FILES */

#define create_proc_file()	do {} while (0)
#define remove_proc_file()	do {} while (0)

#endif				/*CONFIG_USB_GADGET_DEBUG_FILES */

/*-------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------
 * Release the mcf532x specific udc structure
 * it is not stand gadget function
 * it is called when the last reference to the device is removed;
 * it is called from the embedded kobject's release method. 
 * All device structures registered with the core must have a 
 * release method, or the kernel prints out scary complaints
 *-------------------------------------------------------------------------*/
static void
mcf532x_udc_release(struct device *dev)
{
	complete(udc_controller->done);
	kfree(ep_qh_base);
	ep_qh_base = NULL;
	kfree(udc_controller);
	udc_controller = NULL;
}

/******************************************************************
	Internal Structure Build up functions -2
*******************************************************************/
/*------------------------------------------------------------------
 * this func will init resource for globle controller 
 * Return the udc handle on success or Null on failing
 ------------------------------------------------------------------*/
static void *
struct_udc_setup(struct platform_device *pdev)
{
	struct mcf532x_udc *udc = NULL;
	unsigned int tmp_sz = 0;

	udc =
	    (struct mcf532x_udc *) kmalloc(sizeof (struct mcf532x_udc),
					   GFP_KERNEL);
	if (udc == NULL) {
		ERR("malloc udc failed\n");
		goto cleanup;
	}
	/* Zero out the internal USB state structure */
	memset(udc, 0, sizeof (struct mcf532x_udc));

	/* initialized QHs, take care the 2K align */
	tmp_sz = USB_MAX_PIPES * sizeof (struct ep_queue_head);
	udc->ep_qh = (struct ep_queue_head *) KMALLOC_ALIGN(tmp_sz, GFP_KERNEL,
							    2 * 1024,
							    (void **)
							    &ep_qh_base);

	if (!udc->ep_qh) {
		ERR("malloc QHs for udc failed\n");
		goto cleanup;
	}

	/* Initialize ep0 status request structure */
	udc->status_req = container_of(mcf532x_alloc_request(NULL, GFP_KERNEL),
				       struct mcf532x_req, req);
	/* allocate a small amount of memory to get valid address */
	udc->status_req->req.buf = kmalloc(8, GFP_KERNEL);
	udc->status_req->req.dma = virt_to_phys(udc->status_req->req.buf);

	udc->resume_state = USB_STATE_NOTATTACHED;
	udc->usb_state = USB_STATE_POWERED;
	udc->ep0_dir = 0;
	/* initliaze the mcf532x_udc lock */
	spin_lock_init(&udc->lock);

	return udc;

      cleanup:
	kfree(udc);
	return NULL;
}

/*----------------------------------------------------------------
 * set up the mcf532x_ep struct for eps
 * ep0out isnot used so do nothing here
 * ep0in should be taken care 
 * It also link this mcf532x_ep->ep to gadget->ep_list
 *--------------------------------------------------------------*/
static int
struct_ep_setup(void *handle, unsigned char pipe_num)
{
	struct mcf532x_udc *udc = (struct mcf532x_udc *) handle;
	struct mcf532x_ep *ep = get_ep_by_pipe(udc, pipe_num);

	ep->udc = udc;
	strcpy(ep->name, ep_name[pipe_num]);
	ep->ep.name = ep_name[pipe_num];
	ep->ep.ops = &mcf532x_ep_ops;
	ep->stopped = 0;

	/* for ep0: the desc defined here; 
	 * for other eps, gadget layer called ep_enable with defined desc 
	 */
	/* for ep0: maxP defined in desc
	 * for other eps, maxP is set by epautoconfig() called by gadget layer  
	 */
	if (pipe_num == 0) {
		ep->desc = &mcf532x_ep0_desc;
		ep->ep.maxpacket = USB_MAX_CTRL_PAYLOAD;
	} else {
		ep->ep.maxpacket = (unsigned short) ~0;
		ep->desc = NULL;
	}

	/* the queue lists any req for this ep */
	INIT_LIST_HEAD(&ep->queue);

	/* gagdet.ep_list used for ep_autoconfig so no ep0 */
	if (pipe_num != 0)
		list_add_tail(&ep->ep.ep_list, &udc->gadget.ep_list);
	ep->gadget = &udc->gadget;

	return 0;
}

/* Driver probe functions */

static int __init
mcf532x_udc_probe(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	unsigned int i;
	struct otg_transceiver *transceiver = NULL;
	int status = -ENODEV;
	u8 chip_id;
	u8 rev_number;
	int irq_number;

	/*
	 * Check parameters of device resources:
	 */
	if (pdev->num_resources != 2) {
		ERR("invalid num_resources: %i\n", pdev->num_resources);
		return -ENODEV;
	}

	if (pdev->resource[0].flags != IORESOURCE_MEM
	    || pdev->resource[1].flags != IORESOURCE_IRQ) {
		ERR("invalid resource type\n");
		return -ENODEV;
	}

	if (!request_mem_region(pdev->resource[0].start,
				pdev->resource[0].end -
				pdev->resource[0].start + 1, driver_name)) {
		ERR("request_mem_region failed\n");
		return -EBUSY;
	}

	/* Initialize the udc structure including QH member and other member */
	udc_controller = (struct mcf532x_udc *) struct_udc_setup(pdev);
	if (!udc_controller) {
		ERR("udc_controller is NULL \n");
		return -ENOMEM;
	}

	transceiver = otg_get_transceiver();
	if (!transceiver) {
		ERR("No external transceiver!\n");
	}
	udc_controller->transceiver = transceiver;

	/* initialize usb hw reg except for regs for EP, 
	 * leave usbintr reg untouched*/
	dr_controller_setup(udc_controller->ep_qh);

	/*
	 * Simple check of USB Module presence:
	 */
	chip_id = (u8) (MCF_USB0_ID & 0x3F);
	if (chip_id != (u8) ((~MCF_USB0_ID >> 8) & 0x3F)) {
		ERR("is not found!\n");
		goto cleanup;
	}

	rev_number = (u8) (MCF_USB0_ID >> 16 & 0xFF);

	INFO("MCF532x USB Device is found. ID=0x%x Rev=0x%x\n",
	     chip_id, rev_number);

	/* here comes the stand operations for probe
	 * set the mcf532x_udc->gadget.xxx
	 */
	udc_controller->gadget.ops = &mcf532x_gadget_ops;

	/* gadget.ep0 is a pointer */
	udc_controller->gadget.ep0 = &udc_controller->eps[0].ep;

	INIT_LIST_HEAD(&udc_controller->gadget.ep_list);

	udc_controller->gadget.speed = USB_SPEED_UNKNOWN;

	/* name: Identifies the controller hardware type.  */
	udc_controller->gadget.name = driver_name;

	device_initialize(&udc_controller->gadget.dev);

	strcpy(udc_controller->gadget.dev.bus_id, "gadget");

	udc_controller->gadget.dev.release = mcf532x_udc_release;
	udc_controller->gadget.dev.parent = &pdev->dev;

	/* for an EP, the intialization includes: fields in QH, Regs, 
	 * mcf532x_ep struct */
	ep0_dr_and_qh_setup(udc_controller);
	for (i = 0; i < USB_MAX_PIPES; i++) {
		/*because the ep type isnot decide here so
		 * struct_ep_qh_setup() and dr_ep_setup() 
		 * should be called in ep_enable()
		 */
		if (ep_name[i] != NULL)
			/* setup the mcf532x_ep struct and link ep.ep.list
			 * into gadget.ep_list */
			struct_ep_setup(udc_controller, i);
	}

	pullup_disable(udc_controller);
#if	defined(CONFIG_USB_OTG)
	udc_controller->gadget.is_otg = 1;
#endif

	/* USB OTG module IRQ */
	irq_number = platform_get_irq(pdev, 0);
	if (irq_number > 128) {
		status = request_irq(irq_number, mcf532x_udc_irq,
				     SA_INTERRUPT | SA_SHIRQ,
				     driver_name, udc_controller);
		if (status != 0) {
			ERR("can't get irq %ld, err %d\n",
			    pdev->resource[1].start, status);
			goto cleanup;
		} else {
			udc_controller->irq = irq_number;
			irq_number -= 128;

			/*
			 * Set interrupt level:
			 */
			MCF_INTC1_ICR(irq_number) = 0x1;
			/*
			 * Clear Interrupt Mask corresponding bit:
			 */
			MCF_INTC1_CIMR = irq_number;
		}
	} else {
		status = -ENODEV;
		goto cleanup;
	}

	create_proc_file();
	device_add(&udc_controller->gadget.dev);



	return 0;
      cleanup:
	kfree(udc_controller);
	if (transceiver)
		put_device(transceiver->dev);
	release_mem_region(pdev->resource[0].start,
			   pdev->resource[0].end - pdev->resource[0].start + 1);
	return status;

}

/* Driver removal functions
 * Free resources
 * Finish pending transaction
 */
static int __exit
mcf532x_udc_remove(struct device *dev)
{
	int irq_number;
	struct platform_device *pdev = to_platform_device(dev);

	DECLARE_COMPLETION(done);

	if (!udc_controller)
		return -ENODEV;
	udc_controller->done = &done;

	/* DR has been stopped in usb_gadget_unregister_driver() */

	if (udc_controller->transceiver) {
		put_device(udc_controller->transceiver->dev);
		udc_controller->transceiver = NULL;
	}

	/* remove proc */
	remove_proc_file();

	/* free irq */
	irq_number = udc_controller->irq;
	if (irq_number > 128) {
		irq_number -= 128;
		/*
		 * Disable interrupt request:
		 */
		MCF_INTC1_ICR((u32) irq_number) = 0x0;
		/*
		 * Set Interrupt Mask corresponding bit:
		 */
		MCF_INTC1_SIMR = (u8) irq_number;
	}
	free_irq(pdev->resource[1].start, udc_controller);
	udc_controller->irq = 0;

	/* Free allocated memory */
	kfree(udc_controller->status_req->req.buf);
	kfree(udc_controller->status_req);

	/* deinitlaize all ep: strcut */
	/* deinitialize ep0: reg and QH */

	/* free region */
	release_mem_region(pdev->resource[0].start,
			   pdev->resource[0].end - pdev->resource[0].start + 1);

	device_unregister(&udc_controller->gadget.dev);
	/* free udc --wait for the release() finished */
	wait_for_completion(&done);

	return 0;
}

/*-----------------------------------------------------------------
 * Modify Power management attributes
 * Here we stop the DR controller and disable the irq
 -----------------------------------------------------------------*/
static int
mcf532x_udc_suspend(struct device *dev, pm_message_t state)
{
	dr_controller_stop(udc_controller);
	return 0;
}

/*-----------------------------------------------------------------
 * Invoked on USB resume. May be called in_interrupt.
 * Here we start the DR controller and enable the irq
 *-----------------------------------------------------------------*/
static int
mcf532x_udc_resume(struct device *dev)
{
	/* Enable DR irq reg and set controller Run */
	if (udc_controller->stopped) {
		dr_controller_setup(udc_controller->ep_qh);
		dr_controller_run(udc_controller);
	}
	udc_controller->usb_state = USB_STATE_ATTACHED;
	udc_controller->ep0_state = WAIT_FOR_SETUP;
	udc_controller->ep0_dir = 0;

	return 0;
}

/*-------------------------------------------------------------------------
	Register entry point for the peripheral controller driver
--------------------------------------------------------------------------*/
static struct device_driver udc_driver = {
	.name = (char *) driver_name,
	.bus = &platform_bus_type,
	.probe = mcf532x_udc_probe,
	.remove = __exit_p(mcf532x_udc_remove),

	/* these suspend and resume are not related to usb suspend and resume */
	.suspend = mcf532x_udc_suspend,
	.resume = mcf532x_udc_resume,
};

static int __init
udc_init(void)
{
	INFO("%s version %s init \n", driver_desc, DRIVER_VERSION);
	return driver_register(&udc_driver);
}

module_init(udc_init);

static void __exit
udc_exit(void)
{
	driver_unregister(&udc_driver);
}

module_exit(udc_exit);

MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");
