/***************************************************************************
 * ehci-mcf532x.c - EHCI HCD (Host Controller Driver) MCF532x Bus Glue.
 *
 * Andrey Butok
 * Copyright Freescale Semiconductor Inc.  2006
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ***************************************************************************
 * Changes:
 * v0.02	29 September 2006	Andrey Butok
 *		Transceiver support for MCF532x OTG Module is added.
 * v0.01	31 March 2006	Andrey Butok
 *   		Initial Release - developed on uClinux with 2.6.15.6 kernel.
 */

#ifndef CONFIG_M532x
#error "This file is MCF532x bus glue.  CONFIG_M532x must be defined."
#endif

#ifndef CONFIG_USB_EHCI_BIG_ENDIAN
#error "CONFIG_USB_EHCI_BIG_ENDIAN must be defined for MCF532x."
#endif

#ifndef CONFIG_USB_EHCI_ROOT_HUB_TT
#error "CONFIG_USB_EHCI_ROOT_HUB_TT must be defined for MCF532x."
#endif

#include <linux/platform_device.h>

#include <asm/m532xsim.h>
#include <asm/cacheflush.h>

#define MCF_USB_USBMODE_CM(x)			(((x)&0x00000003)<<0)
#define MCF_USB_USBMODE_CM_HOST			(0x00000003)
#define MCF_USB_USBMODE_ES			(0x00000004)

/*-------------------------------------------------------------------------*/

/*
 * ehci_mcf532x_setup() called during probe() after chip reset completes
 */
static int
ehci_mcf532x_setup(struct usb_hcd *hcd)
{
	struct ehci_hcd *ehci = hcd_to_ehci(hcd);
	int retval;

	ehci->caps = hcd->regs;
	ehci->regs = hcd->regs + HC_LENGTH(readl(&ehci->caps->hc_capbase));
	dbg_hcs_params(ehci, "reset");
	dbg_hcc_params(ehci, "reset");

	/*
	 *  cache this readonly data; minimize chip reads.
	 */
	ehci->hcs_params = readl(&ehci->caps->hcs_params);

	retval = ehci_halt(ehci);
	if (retval)
		return retval;

	/*
	 * data structure init.
	 */
	retval = ehci_init(hcd);
	if (retval)
		return retval;
	ehci->is_tdi_rh_tt = 1;

	return retval;
}

static irqreturn_t
ehci_mcf532x_irq(struct usb_hcd *hcd, struct pt_regs *regs)
{
	irqreturn_t status = IRQ_NONE;
	if ((readl((u32 *) (((u8 *) hcd->regs) + 0xA8)) &
	     MCF_USB_USBMODE_CM(0xF))
	    == MCF_USB_USBMODE_CM_HOST) {
		flush_cache_all();	/* Cache issue workaround */
		status = ehci_irq(hcd, regs);
	}
	return status;
}

static const struct hc_driver ehci_mcf532x_hc_driver = {
	.description = hcd_name,
	.product_desc = "EHCI Host Controller",
	.hcd_priv_size = sizeof (struct ehci_hcd),

	/*
	 * generic hardware linkage
	 */
	.irq = ehci_mcf532x_irq,
	.flags = HCD_MEMORY | HCD_USB2,

	/*
	 * basic lifecycle operations
	 */
	.reset = ehci_mcf532x_setup,
	.start = ehci_run,
	.stop = ehci_stop,

	/*
	 * managing i/o requests and associated device resources
	 */
	.urb_enqueue = ehci_urb_enqueue,
	.urb_dequeue = ehci_urb_dequeue,
	.endpoint_disable = ehci_endpoint_disable,

	/*
	 * scheduling support
	 */
	.get_frame_number = ehci_get_frame,

	/*
	 * root hub support
	 */
	.hub_status_data = ehci_hub_status_data,
	.hub_control = ehci_hub_control,
	.bus_suspend = ehci_bus_suspend,
	.bus_resume = ehci_bus_resume,
};

/*-------------------------------------------------------------------------*/

/**
 * ehci_hcd_mcf532x_probe - initialize MCF532x-based HCDs
 * @hcd: USB Host Controller being initialized.
 * @pdev: USB device on platform bus. The USB host controller
 *        driver will be assigned to it.
 * Context: !in_interrupt()
 *
 * Allocates basic resources for this USB host controller,
 */
int
ehci_hcd_mcf532x_probe(const struct hc_driver *driver,
		       struct platform_device *pdev, struct device *dev)
{
	int retval = 0;
	u8 irq_number;
	u8 chip_id;
	int rev_number;
	int timeout;
	struct usb_hcd *hcd = 0;
	struct ehci_hcd *ehci = 0;

	/*
	 * Check parameters of device resources:
	 */
	if (pdev->num_resources != 2) {
		printk(KERN_ERR "hcd probe: invalid num_resources: %i\n",
		       pdev->num_resources);
		return -ENODEV;
	}

	if (pdev->resource[0].flags != IORESOURCE_MEM
	    || pdev->resource[1].flags != IORESOURCE_IRQ) {
		printk(KERN_ERR "hcd probe: invalid resource type\n");
		return -ENODEV;
	}

	/*
	 * Create and initialize HCD structure:
	 */
	hcd = usb_create_hcd(driver, &pdev->dev, pdev->dev.bus_id);
	if (!hcd) {
		retval = -ENOMEM;
		goto err0;
	}

	ehci = hcd_to_ehci(hcd);
	dev_set_drvdata(dev, ehci);	/* required for supsend/resume */

	/*
	 * Request resources:
	 */
	hcd->rsrc_start = pdev->resource[0].start;
	hcd->rsrc_len = pdev->resource[0].end - pdev->resource[0].start + 1;

	if (!request_mem_region(hcd->rsrc_start, hcd->rsrc_len, hcd_name)) {
		dev_dbg(&pdev->dev, "request_mem_region failed\n");
		retval = -EBUSY;
		goto err1;
	}

	/*
	 * Init pointer to EHCI register
	 */
	hcd->regs = (void __iomem *) ((u32) (hcd->rsrc_start) + 0x100);

	/*
	 * Reset controller.
	 * USBCMD = MCF_USB_USBCMD_RST
	 */
	writel((0x00000002), (u32 *) (((u8 *) hcd->regs) + 0x40));
	/* Wait for reset complete */
	timeout = 500;
	while (timeout-- && (readl((u32 *) (((u8 *) hcd->regs) + 0x40)) & 0x2))
		udelay(1);
	if (timeout <= 0)
		printk("%s - USBCMD_RST never clear. Timeout is %d \n",
		       __FUNCTION__, timeout);

	/*
	 * Set "Host mode" and "Big endian byte ordering" for the transfer buffers:
	 * USBMODE = MCF_USB_USBMODE_ES_BIGENDIAN
	 *                              | MCF_USB_USBMODE_CM_HOST
	 */
	writel((MCF_USB_USBMODE_CM_HOST | MCF_USB_USBMODE_ES),
	       (u32 *) (((u8 *) hcd->regs) + 0xA8));

	/*
	 * Simple check of USB Module presence:
	 */
	chip_id = (u8) (readl((u32 *) ((u32) hcd->rsrc_start)) & 0x3F);
	if (chip_id !=
	    (u8) ((~readl((u32 *) ((u32) hcd->rsrc_start)) >> 8) & 0x3F)) {
		retval = -ENODEV;
		goto err2;
	}

	rev_number = (u8) (readl((u32 *) ((u32) hcd->rsrc_start)) >> 16 & 0xFF);

	pr_info("MCF532x USB EHCI: is found. ID=0x%x Rev=0x%x\n",
		chip_id, rev_number);

	/*
	 * Finish generic HCD structure initialization and register:
	 */
	irq_number = platform_get_irq(pdev, 0);
	if (irq_number > 128) {
		retval = usb_add_hcd(hcd, irq_number, SA_INTERRUPT | SA_SHIRQ);
		if (retval == 0) {
			irq_number -= 128;;
			/*
			 * Set interrupt level:
			 */
			MCF_INTC1_ICR((u32) irq_number) = 0x1;
			/*
			 * Clear Interrupt Mask corresponding bit:
			 */
			MCF_INTC1_CIMR = (u8) irq_number;

			if ((readl((u32 *) (((u8 *) hcd->regs) + 0x24)) & 0x180)
			    == 0x180) {
				struct ehci_hcd *ehci = hcd_to_ehci(hcd);
#ifdef CONFIG_USB_OTG
				ehci_to_hcd(ehci)->self.otg_port = 1;
				/* default/minimum OTG power budget:  8 mA */
				/* ehci_to_hcd(ehci)->power_budget = 8; */
#endif

				ehci->transceiver = otg_get_transceiver();
				if (ehci->transceiver) {
					int status =
					    otg_set_host(ehci->transceiver,
							 &ehci_to_hcd(ehci)->
							 self);
					pr_info
					    ("ehci-mcf532x: init %s transceiver, status %d\n",
					     ehci->transceiver->label, status);
					if (status) {
						if (ehci->transceiver)
							put_device(ehci->
								   transceiver->
								   dev);
						return status;
					}
				} else {
					pr_info
					    ("ehci-mcf532x: can't find transceiver!\n");
					retval = -ENODEV;
				}
			}

			if (retval == 0)
				return retval;
		}
	} else
		retval = -ENODEV;
      err2:
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
      err1:
	usb_put_hcd(hcd);
      err0:
	return retval;
}

/**
 * ehci_hcd_mcf532x_remove - shutdown processing for MCF532x-based HCDs
 * @hcd: USB Host Controller being removed.
 * @pdev: USB device on platform bus. The USB host controller driver
 *        is assigned to it.
 * Context: !in_interrupt()
 *
 * Reverses the effect of ehci_hcd_mcf532x_probe().
 * It is always called from a thread context, normally "rmmod" or
 * something similar.
 *
 */
void
ehci_hcd_mcf532x_remove(struct usb_hcd *hcd, struct platform_device *pdev)
{
	int irq_number = hcd->irq;
	if (irq_number > 128) {
		irq_number -= 128;
		/*
		 * Disable interrupt request:
		 */
		MCF_INTC1_ICR((u32) irq_number) = 0x0;
		/*
		 * Set Interrupt Mask corresponding bit:
		 */
		MCF_INTC1_SIMR = (u8) irq_number;
	}

	usb_remove_hcd(hcd);
	release_mem_region(hcd->rsrc_start, hcd->rsrc_len);
	usb_put_hcd(hcd);
}

/*-------------------------------------------------------------------------*/

static int
ehci_hcd_mcf532x_drv_probe(struct device *dev)
{
	int result = ehci_hcd_mcf532x_probe(&ehci_mcf532x_hc_driver,
					    to_platform_device(dev), dev);
	return result;
}

static int
ehci_hcd_mcf532x_drv_remove(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct usb_hcd *hcd = dev_get_drvdata(dev);

	ehci_hcd_mcf532x_remove(hcd, pdev);
	return 0;
}

/*-------------------------------------------------------------------------*/

/* This following codes are used by USB OTG module*/
volatile static struct ehci_regs usb_ehci_regs;

/*
 * This function is used only when host off,first free irq,
 * then stop Host controller,and save the values of USB Host registers
 */
static int
ehci_hcd_mcf532x_suspend(struct device *dev, pm_message_t state)
{
	struct ehci_hcd *ehci = dev_get_drvdata(dev);
	struct usb_hcd *hcd = ehci_to_hcd(ehci);
	int retval;

	free_irq(hcd->irq, hcd);

	retval = readl(&ehci->regs->command);
	retval &= ~CMD_RUN;
	writel(retval, &ehci->regs->command);

	memcpy((void *) &usb_ehci_regs, ehci->regs, sizeof (struct ehci_regs));

	usb_ehci_regs.port_status[0] &=
	    cpu_to_hc32(~(PORT_PEC | PORT_OCC | PORT_CSC));

	return 0;
}

/*
 * This function is used only when host on.
 * First,set the USB controller's mode to USB HOST,
 * then,restore the values of USB Host registers,request irq,
 * last,run the USB controller with USB Host mode.
 */
static int
ehci_hcd_mcf532x_resume(struct device *dev)
{
	struct ehci_hcd *ehci = dev_get_drvdata(dev);
	struct usb_hcd *hcd = ehci_to_hcd(ehci);
	int retval;

	writel((MCF_USB_USBMODE_CM_HOST | MCF_USB_USBMODE_ES),
	       (u32 *) (((u8 *) hcd->regs) + 0xA8));

	memcpy(ehci->regs, (void *) &usb_ehci_regs, sizeof (struct ehci_regs));

	retval = request_irq(hcd->irq, usb_hcd_irq, SA_INTERRUPT | SA_SHIRQ,
			     hcd->driver->description, hcd);

	if (retval != 0)
		return retval;

	retval = readl(&ehci->regs->command);
	retval |= CMD_RUN;
	writel(retval, &ehci->regs->command);

	return 0;
}

/*-------------------------------------------------------------------------*/

/*
 * Device driver definition to register with the Platform bus
 */
static struct device_driver ehci_mcf532x_driver = {
	.name = "ehci",
	.bus = &platform_bus_type,
	.probe = ehci_hcd_mcf532x_drv_probe,
	.remove = ehci_hcd_mcf532x_drv_remove,

	.suspend = ehci_hcd_mcf532x_suspend,
	.resume = ehci_hcd_mcf532x_resume,
};

static int __init
ehci_hcd_mcf532x_init(void)
{
	if (usb_disabled())
		return -ENODEV;

	pr_debug("%s: block sizes: qh %Zd qtd %Zd itd %Zd sitd %Zd\n",
		 hcd_name,
		 sizeof (struct ehci_qh), sizeof (struct ehci_qtd),
		 sizeof (struct ehci_itd), sizeof (struct ehci_sitd));

	return driver_register(&ehci_mcf532x_driver);
}

module_init(ehci_hcd_mcf532x_init);

static void __exit
ehci_hcd_mcf532x_cleanup(void)
{
	driver_unregister(&ehci_mcf532x_driver);
}

module_exit(ehci_hcd_mcf532x_cleanup);
