/*
 *  Yaroslav Vinogradov yaroslav.vinogradov@freescale.com
 *  Andrey Butok
 * Copyright (c) Freescale Semiconductor, Inc. 2006
 *
 * USB ULPI PHY driver for M5329
 *
 * Based on code for MPC from:
 *  Leo Li<LeoLi@freescale.com>
 *  Jerry Huang<Chang-Ming.Huang@freescale.com>
 *
 *  Initialization based on code from Shlomi Gridish.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the  GNU General Public License along
 * with this program; if not, write  to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 ***************************************************************************
 * Changes:
 *   v0.2	20 October 2006		Andrey Butok
 *		Issues fixing and some changes.
 *   v0.1	September 2006		Yaroslav Vinogradov
 *		Initial version.
 *
 */

#if 0
#define DEBUG 	1
#define VERBOSE 	1
#endif

#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/smp_lock.h>
#include <linux/proc_fs.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/reboot.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/usb.h>
#include <linux/device.h>
#include <linux/usb_ch9.h>
#include <linux/usb_gadget.h>
#include <linux/workqueue.h>
#include <linux/time.h>
#include <linux/platform_device.h>

#include <asm/m532xsim.h>

#include "fsl_otg.h"

#if 0
#define DEBUG 	1
#define VERBOSE 	1
#endif

#ifdef DEBUG
#define DBG(fmt, args...) 	printk( "[%s]  " fmt , \
				__FUNCTION__, ## args)
#else
#define DBG(fmt, args...)	do{}while(0)
#endif

#ifdef VERBOSE
#define VDBG		DBG
#else
#define VDBG(stuff...)	do{}while(0)
#endif




#define CONFIG_USB_OTG_DEBUG_FILES
#define DRIVER_VERSION "Revision: 0.2"
#define DRIVER_AUTHOR "Yaroslav Vinogradov"
#define DRIVER_DESC "Freescale USB OTG Driver"
#define DRIVER_INFO DRIVER_VERSION " " DRIVER_DESC

MODULE_DESCRIPTION("M532x USB OTG Controller Driver");

static const char otg_dr_name[] = "mcf532x-ulpi";

/*#define HA_DATA_PULSE 1*/

static struct fsl_otg *fsl_otg_dev = NULL;
static int srp_wait_done;


/* Driver specific timers */
struct fsl_otg_timer *b_data_pulse_tmr, *b_vbus_pulse_tmr, *b_srp_fail_tmr,
	*b_srp_wait_tmr, *a_wait_enum_tmr;

static struct list_head active_timers;

static struct fsl_otg_config fsl_otg_initdata = {
	.otg_port = 1,
};

/* Routines to access transceiver ULPI registers */
u8 view_ulpi(u8 addr)
{
	u32 temp;

	temp = 0x40000000 | (addr << 16);
	MCF_USB0_ULPI_VIEWPORT = temp;

	udelay(10);/*1000*/

	temp = MCF_USB0_ULPI_VIEWPORT;
	while (temp & 0x40000000) {
		temp = MCF_USB0_ULPI_VIEWPORT;
	}
	return (temp & 0x0000ff00) >> 8;
}

int write_ulpi(u8 addr, u8 data)
{
	u32 temp;

	temp = 0x60000000 | (addr << 16) | data;
	MCF_USB0_ULPI_VIEWPORT = temp;
	return 0;
}

/* prototype declaration */
void fsl_otg_add_timer(void *timer);
void fsl_otg_del_timer(void *timer);

/* -------------------------------------------------------------*/
/* Operations that will be called from OTG Finite State Machine */

/* Charge vbus for vbus pulsing in SRP */
void fsl_otg_chrg_vbus(int on)
{
	VDBG(" %d\n", on);
	if (on) {
		MCF_USB0_OTGSC = (MCF_USB0_OTGSC & ~OTGSC_INTSTS_MASK & ~OTGSC_CTRL_VBUS_DISCHARGE) | OTGSC_CTRL_VBUS_CHARGE;
	} else {
		MCF_USB0_OTGSC = MCF_USB0_OTGSC & ~OTGSC_INTSTS_MASK & ~OTGSC_CTRL_VBUS_CHARGE;
	}
}

/* Discharge vbus through a resistor to ground */
void fsl_otg_dischrg_vbus(int on)
{
	VDBG(" %d\n", on);
	if (on) {
		MCF_USB0_OTGSC = (MCF_USB0_OTGSC & ~OTGSC_INTSTS_MASK) | OTGSC_CTRL_VBUS_DISCHARGE;
	} else {
		MCF_USB0_OTGSC = MCF_USB0_OTGSC & ~OTGSC_INTSTS_MASK & ~OTGSC_CTRL_VBUS_DISCHARGE;
	}
}

/* A-device driver vbus, controlled through PP bit in PORTSC */
void fsl_otg_drv_vbus(int on)
{
//	u8 ulpi_sts=view_ulpi(0xA);//DM
	VDBG(" %d\n", on);

	if (on) {
		MCF_USB0_PORTSC = (MCF_USB0_PORTSC & ~PORTSC_W1C_BITS) | PORTSC_PORT_POWER;
//		write_ulpi(0xA, ulpi_sts&(~0x40));//DM FIXME
	} else {
		MCF_USB0_PORTSC = MCF_USB0_PORTSC & ~PORTSC_W1C_BITS & ~PORTSC_PORT_POWER;
//		write_ulpi(0xA, ulpi_sts|0x40);//DM FIXME
	}
}

/* Pull-up D+, signalling connect by periperal. Also used in
 * data-line pulsing in SRP */
void fsl_otg_loc_conn(int on)
{
	VDBG(" %d\n", on);
	if (on) {
		MCF_USB0_OTGSC = (MCF_USB0_OTGSC & ~OTGSC_INTSTS_MASK) | OTGSC_CTRL_DATA_PULSING;
	} else {
		MCF_USB0_OTGSC = MCF_USB0_OTGSC & ~OTGSC_INTSTS_MASK & ~OTGSC_CTRL_DATA_PULSING;
	}
}

/* Generate SOF by host.  This is controlled through suspend/resume the
 * port.  In host mode, controller will automatically send SOF.
 * Suspend will block the data on the port. */
void fsl_otg_loc_sof(int on)
{
	u32 tmpval;
	VDBG(" %d\n", on);
	tmpval = MCF_USB0_PORTSC & ~PORTSC_W1C_BITS;
	if (on) {
		tmpval |= PORTSC_PORT_FORCE_RESUME;
	} else {
		tmpval |= PORTSC_PORT_SUSPEND;
	}
	MCF_USB0_PORTSC = tmpval;
}

/* Start SRP pulsing by data-line pulsing, followed with v-bus pulsing. */
void fsl_otg_start_pulse(void)
{
	VDBG("\n");
	srp_wait_done = 0;
#ifdef HA_DATA_PULSE
	MCF_USB0_OTGSC = (MCF_USB0_OTGSC & ~OTGSC_INTSTS_MASK) | OTGSC_HA_DATA_PULSE;
#else
	fsl_otg_loc_conn(1);
#endif

	fsl_otg_add_timer(b_data_pulse_tmr);
}

void fsl_otg_pulse_vbus(void);

void b_data_pulse_end(unsigned long foo)
{
#ifdef HA_DATA_PULSE
#else
	fsl_otg_loc_conn(0);
#endif

	/* Do VBUS pulse after data pulse */
	fsl_otg_pulse_vbus();
}

void fsl_otg_pulse_vbus(void)
{
	srp_wait_done = 0;
	fsl_otg_chrg_vbus(1);
	/* start the timer to end vbus charge */
	fsl_otg_add_timer(b_vbus_pulse_tmr);
}

void b_vbus_pulse_end(unsigned long foo)
{
	fsl_otg_chrg_vbus(0);

	/* As USB3300 using the same a_sess_vld and b_sess_vld voltage
	 * we need to discharge the bus for a while to distinguish
	 * residual voltage of vbus pulsing and A device pull up */
	fsl_otg_dischrg_vbus(1);
	fsl_otg_add_timer(b_srp_wait_tmr);
}

void b_srp_end(unsigned long foo)
{
	fsl_otg_dischrg_vbus(0);
	srp_wait_done = 1;

	if ((fsl_otg_dev->otg.state == OTG_STATE_B_SRP_INIT) &&
	    fsl_otg_dev->fsm.b_sess_vld)
		fsl_otg_dev->fsm.b_srp_done = 1;
}

/* Workaround for a_host suspending too fast.  When a_bus_req=0,
 * a_host will start by SRP.  It needs to set b_hnp_enable before
 * actually suspending to start HNP */
void a_wait_enum(unsigned long foo)
{
	VDBG("\n");
	if (!fsl_otg_dev->otg.host->b_hnp_enable)
		fsl_otg_add_timer(a_wait_enum_tmr);
	else
		otg_fsm_statemachine(&fsl_otg_dev->fsm);
}

/* ------------------------------------------------------*/

/* The timeout callback function to set time out bit */
void set_tmout(unsigned long indicator)
{
	*(int *)indicator = 1;
}

/* Initialize timers */
void fsl_otg_init_timers(struct otg_fsm *fsm)
{
	/* FSM used timers */
	fsm->a_wait_vrise_tmr = otg_timer_initializer(&set_tmout, TA_WAIT_VRISE,
				(unsigned long)&fsm->a_wait_vrise_tmout);
	fsm->a_wait_bcon_tmr = otg_timer_initializer(&set_tmout, TA_WAIT_BCON,
				(unsigned long)&fsm->a_wait_bcon_tmout);
	fsm->a_aidl_bdis_tmr = otg_timer_initializer(&set_tmout, TA_AIDL_BDIS,
				(unsigned long)&fsm->a_aidl_bdis_tmout);
	fsm->b_ase0_brst_tmr = otg_timer_initializer(&set_tmout, TB_ASE0_BRST,
				(unsigned long)&fsm->b_ase0_brst_tmout);
	fsm->b_se0_srp_tmr = otg_timer_initializer(&set_tmout, TB_SE0_SRP,
				(unsigned long)&fsm->b_se0_srp);
	fsm->b_srp_fail_tmr = otg_timer_initializer(&set_tmout, TB_SRP_FAIL,
				(unsigned long)&fsm->b_srp_done);
	fsm->a_wait_enum_tmr = otg_timer_initializer(&a_wait_enum, 10,
				(unsigned long)&fsm);

	/* device driver used timers */
	b_srp_wait_tmr = otg_timer_initializer(&b_srp_end, TB_SRP_WAIT, 0);
	b_data_pulse_tmr = otg_timer_initializer(&b_data_pulse_end, 
				TB_DATA_PLS, 0);
	b_vbus_pulse_tmr = otg_timer_initializer(&b_vbus_pulse_end,
				TB_VBUS_PLS, 0);

}

/* Add timer to timer list */
void fsl_otg_add_timer(void *gtimer)
{
	struct fsl_otg_timer *timer = (struct fsl_otg_timer *)gtimer;
	struct fsl_otg_timer *tmp_timer;

	/* Check if the timer is already in the active list, 
	 * if so update timer count */
	list_for_each_entry(tmp_timer, &active_timers, list)
	    if (tmp_timer == timer) {
		timer->count = timer->expires;
		return;
	}
	timer->count = timer->expires;
	list_add_tail(&timer->list, &active_timers);
}

/* Remove timer from the timer list; clear timeout status */
void fsl_otg_del_timer(void *gtimer)
{
	struct fsl_otg_timer *timer = (struct fsl_otg_timer *)gtimer;
	struct fsl_otg_timer *tmp_timer, *del_tmp;

	list_for_each_entry_safe(tmp_timer, del_tmp, &active_timers, list)
		if (tmp_timer == timer)
			list_del(&timer->list);
}

/* Reduce timer count by 1, and find timeout conditions.
 * Called by fsl_otg 1ms timer interrupt
 */
int fsl_otg_tick_timer(void)
{
	struct fsl_otg_timer *tmp_timer, *del_tmp;
	int expired = 0;

	list_for_each_entry_safe(tmp_timer, del_tmp, &active_timers, list) {
		tmp_timer->count--;
		/* check if timer expires */
		if (!tmp_timer->count) {
			list_del(&tmp_timer->list);
			tmp_timer->function(tmp_timer->data);
			expired = 1;
		}
	}

	return expired;
}

/* Reset controller, not reset the bus */
void otg_reset_controller(void)
{
	u32 command;
	int timeout;

	VDBG("\n");

	command = MCF_USB0_USBCMD;
	command |= MCF_USB_USBCMD_RST;
	MCF_USB0_USBCMD = command;

	/* Wait reset completed */
	timeout = 500;
	while (timeout-- && (MCF_USB0_USBCMD&0x2))
		udelay(1);
	if (timeout <=0)
		ERR("%s - USBCMD_RST never clear. Timeout is %d \n",
			__FUNCTION__, timeout);
}

/* Call suspend/resume routines in host driver */
int fsl_otg_start_host(struct otg_fsm *fsm, int on)
{
	struct otg_transceiver *xceiv = fsm->transceiver;
	struct device *dev = xceiv->host->controller;
	struct fsl_otg *otg_dev = container_of(xceiv, struct fsl_otg, otg);
	u32 retval = 0;

	VDBG(" %p\n", fsm);

#ifdef DEBUG
	if (otg_dev->host_working) {
		printk("host working!\n");
	} else {
		printk("host is not working!\n");
	}
#endif
	/* Update a_vbus_vld state as a_vbus_vld int is disabled
	 * in device mode */
/*	otgsc = MCF_USB0_OTGSC;
	fsm->a_vbus_vld = (MCF_USB0_OTGSC & OTGSC_STS_A_VBUS_VALID) ? 1 : 0;*/
/*	reg = view_ulpi(0x13); // read the USB interrupt status register from ULPI PHY
	fsm->a_vbus_vld = (reg & 0x1) ? 1:0;*/

	if (!xceiv->host) {
#ifdef DEBUG
		printk("no dev!\n");
#endif
		return -ENODEV;
	}

	if (on) {
		/*start fsl usb host controller */
		if (otg_dev->host_working)
			goto end;
		else {
			otg_reset_controller();
			VDBG("host on......\n");
			if (dev->driver->resume) {
				retval = dev->driver->resume(dev/*, 0*/);
				if (fsm->id) {
					/* default-b */
					fsl_otg_drv_vbus(1);
					/* Workaround: b_host can't driver
					 * vbus, but PP in PORTSC needs to
					 * be 1 for host to work.
					 * So we set drv_vbus bit in
					 * transceiver to 0 thru ULPI. */
					write_ulpi(0x0c, 0x20);
				}
			}
			otg_dev->host_working = 1;
		}
	} else {
		/*stop fsl usb host controller */
		if (!otg_dev->host_working)
			goto end;
		else {
			VDBG("host off......\n");
			if (dev && dev->driver) {
				retval = dev->driver->suspend(dev, PMSG_SUSPEND/*, 0*/);
				if (fsm->id)
					/* default-b */
					fsl_otg_drv_vbus(0);
			}
			otg_dev->host_working = 0;
		}
	}
end:
	return retval;
}

/* Call suspend and resume function in udc driver
 * to stop and start udc driver.
 */
int fsl_otg_start_gadget(struct otg_fsm *fsm, int on)
{
	struct otg_transceiver *xceiv = fsm->transceiver;
	struct device *dev = &xceiv->gadget->dev;

	if (!xceiv->gadget)
		return -ENODEV;

	VDBG("starting gadget %s \n", on ? "on" : "off");

	if (on)
	{
		dev->driver->resume(dev/*, 0*/);

#ifdef CONFIG_USB_OTG_ULPI_FS
		MCF_USB0_PORTSC|=MCF_USB_PORTSC_PFSC; /* Set FS only */
#endif
	}
	else
		dev->driver->suspend(dev, PMSG_SUSPEND /* 0, 0 */);

	return 0;
}

static void fsl_otg_enable(struct otg_transceiver *otg_p)
{
	struct fsl_otg *otg_dev = container_of(otg_p, struct fsl_otg, otg);
	struct otg_fsm *fsm = &(otg_dev)->fsm;
	u32 otg_sc;

	/* enable OTG interrupt */
	otg_sc = MCF_USB0_OTGSC;
	otg_sc |= OTGSC_INTERRUPT_ENABLE_BITS_MASK;
	otg_sc &= ~OTGSC_CTRL_VBUS_DISCHARGE;
	MCF_USB0_OTGSC = otg_sc;


	fsm->id = (otg_sc & OTGSC_STS_USB_ID) ? 1 : 0;

	if (fsm->id) {
		otg_p->state = OTG_STATE_UNDEFINED;
	} else {
		otg_p->state = OTG_STATE_A_IDLE;
	}

	otg_p->default_a = (fsm->id == 0);
	otg_p->host->is_b_host = fsm->id;
	otg_p->gadget->is_a_peripheral = !fsm->id;

	fsm->a_vbus_vld =1;

	fsm->b_sess_vld = (otg_sc & OTGSC_STS_B_SESSION_VALID) ? 1 : 0;
	fsm->a_sess_vld = (otg_sc & OTGSC_STS_A_SESSION_VALID) ? 1 : 0;
	fsm->b_sess_end = (otg_sc & OTGSC_STS_B_SESSION_END) ? 1 : 0;
}

/* Called by initialization code of host driver.  Register host controller
 * to the OTG.  Suspend host for OTG role detection. */
static int fsl_otg_set_host(struct otg_transceiver *otg_p, struct usb_bus *host)
{
	struct fsl_otg *otg_dev = container_of(otg_p, struct fsl_otg, otg);
	struct device *dev = host->controller;

	if (!otg_p || otg_dev != fsl_otg_dev)
		return -ENODEV;

	otg_p->host = host;

	otg_dev->fsm.a_bus_drop = 0;
	otg_dev->fsm.a_bus_req = 1;

	otg_p->host->otg_port = fsl_otg_initdata.otg_port;

	VDBG("host off......\n");
	if (dev && dev->driver)
		dev->driver->suspend(dev, PMSG_SUSPEND /* 0, 0*/);
	otg_dev->host_working = 0;

	if (otg_p->gadget) {
		fsl_otg_enable(otg_p);
		otg_fsm_statemachine(&otg_dev->fsm);
	}
	return 0;
}

/* Called by initialization code of udc.  Register udc to OTG.*/
static int fsl_otg_set_peripheral(struct otg_transceiver *otg_p,
				  struct usb_gadget *gadget)
{
	struct fsl_otg *otg_dev = container_of(otg_p, struct fsl_otg, otg);

	VDBG("\n");

	if (!otg_p || otg_dev != fsl_otg_dev)
		return -ENODEV;

	if (!gadget) {
		if (!otg_dev->otg.default_a)
			otg_p->gadget->ops->vbus_draw(otg_p->gadget, 0);
		usb_gadget_vbus_disconnect(otg_dev->otg.gadget);
		otg_dev->otg.gadget = 0;
		otg_dev->fsm.b_bus_req = 0;
		otg_fsm_statemachine(&otg_dev->fsm);
		return 0;
	}

	otg_p->gadget = gadget;

	otg_dev->fsm.b_bus_req = 1;

	if (otg_p->host) {
		fsl_otg_enable(otg_p);
		otg_fsm_statemachine(&otg_dev->fsm);
	}
	return 0;
}

/* Set OTG port power,only for B-device */
static int fsl_otg_set_power(struct otg_transceiver *otg_p, unsigned mA)
{
	if (!fsl_otg_dev)
		return -ENODEV;
	if (otg_p->state == OTG_STATE_B_PERIPHERAL)
		printk("FSL OTG:Draw %d mA\n", mA);

	return 0;
}

/* B-device start SRP */
static int fsl_otg_start_srp(struct otg_transceiver *otg_p)
{
	struct fsl_otg *otg_dev = container_of(otg_p, struct fsl_otg, otg);

	if (!otg_p || otg_dev != fsl_otg_dev
	    || otg_p->state != OTG_STATE_B_IDLE)
		return -ENODEV;

	otg_dev->fsm.b_bus_req = 1;
	otg_fsm_statemachine(&otg_dev->fsm);

	return 0;
}

/* A_host suspend will call this function to start hnp */
static int fsl_otg_start_hnp(struct otg_transceiver *otg_p)
{
	struct fsl_otg *otg_dev = container_of(otg_p, struct fsl_otg, otg);

	if (!otg_p || otg_dev != fsl_otg_dev)
		return -ENODEV;

	VDBG("start_hnp.............\n");
	/* clear a_bus_req to enter a_suspend state */
	otg_dev->fsm.a_bus_req = 0;
	otg_fsm_statemachine(&otg_dev->fsm);

	return 0;
}

//static u8 reg_tmp;//DM
/* Interrupt handler.  OTG/host/peripheral share the same int line.
 * OTG driver clears OTGSC interrupts and leaves USB interrupts
 * intact.  It needs to have knowledge of some USB interrupts
 * such as port change. */

irqreturn_t fsl_otg_isr(int irq, void *dev_id, struct pt_regs * regs)
{
	struct otg_fsm *fsm = &((struct fsl_otg *)dev_id)->fsm;
	struct otg_transceiver *otg = &((struct fsl_otg *)dev_id)->otg;
	u32 otg_int_src, usb_int_src, otg_sc;
	int trigger = 0;
	int tmp;

	usb_int_src = MCF_USB0_USBSTS;
	otg_sc = MCF_USB0_OTGSC;
	otg_int_src = otg_sc & OTGSC_INTSTS_MASK & (otg_sc >> 8);

	/* Only clear otg interrupts */
	MCF_USB0_OTGSC |= otg_sc & OTGSC_INTSTS_MASK;

	/*FIXME: ID change not generate when init to 0 */
//DM	fsm->id = (otg_sc & OTGSC_STS_USB_ID) ? 1 : 0;
//DM	otg->default_a = (fsm->id == 0);

	/* process OTG interrupts */
	if (otg_int_src) {
		if (otg_int_src & OTGSC_INTSTS_1MS) {
			u32 ccm;
			u8 reg;
			trigger = fsl_otg_tick_timer();
/*
			reg = view_ulpi(0x13); // read the USB interrupt status register from ULPI PHY

			if(reg_tmp!=reg)
			{
				//printk("!!!!MCF_CCM_UOCSR=0x%x \n",ccm);//DM
				u8 ulpi_sts;
				printk("!!!!ULPI STS=0x%x \n",reg);//DM
				printk("!!!!MCF_USB0_OTGSC=0x%x \n",MCF_USB0_OTGSC);//DM
				ulpi_sts=view_ulpi(0xA);//DM
				printk("!!!!ULPI Control=0x%x \n",ulpi_sts);//DM
			}
			reg_tmp=reg;*/
		}
		if (otg_int_src & OTGSC_INTSTS_USB_ID) {
			fsm->id = (otg_sc & OTGSC_STS_USB_ID) ? 1 : 0;
			otg->default_a = (fsm->id == 0);
			if (otg->host)
				otg->host->is_b_host = fsm->id;
			if (otg->gadget)
				otg->gadget->is_a_peripheral = !fsm->id;
			trigger = 1;
			VDBG("ID int (ID is %d)\n", fsm->id);
		}
		if (otg_int_src & OTGSC_INTSTS_DATA_PULSING) {
			fsm->a_srp_det = 1;
			trigger = 1;
			VDBG("!!!!Data pulse int \n");
		}
		if (otg_int_src & OTGSC_INTSTS_A_SESSION_VALID) {
			fsm->a_sess_vld =
			    (otg_sc & OTGSC_STS_A_SESSION_VALID) ? 1 : 0;
			/* detect VBUS pulsing */
			if ((fsm->transceiver->state == OTG_STATE_A_IDLE)
			    && fsm->a_sess_vld)
				fsm->a_srp_det = 1;
			trigger = 1;
			VDBG("!!!a_sess_vld int state=%d \n", fsm->a_sess_vld);
		}
		if (otg_int_src & OTGSC_INTSTS_A_VBUS_VALID) {
			fsm->a_vbus_vld =(otg_sc & OTGSC_STS_A_VBUS_VALID) ? 1 : 0;
			trigger = 1;
			VDBG("!!!a_vbus_vld int state=%d \n", fsm->a_vbus_vld);

#if 1
		if ((fsm->a_vbus_vld == 0) && (fsm->id == 0)) {
			if (fsm->transceiver->state != OTG_STATE_A_WAIT_VFALL)
				/* Warning insted of transfer to Error state. */
				printk(KERN_WARNING "ulpi: The USB Device is sinking too much current!!\n");
			fsm->a_vbus_vld = 1;
		}
#endif


		}
		if (otg_int_src & OTGSC_INTSTS_B_SESSION_VALID) {
			fsm->b_sess_vld =
			    (otg_sc & OTGSC_STS_B_SESSION_VALID) ? 1 : 0;
			trigger = 1;
			/* SRP done */
			if ((fsl_otg_dev->otg.state == OTG_STATE_B_SRP_INIT) &&
			    fsm->b_sess_vld && srp_wait_done)
				fsm->b_srp_done = 1;
			VDBG("!!!!b_sess_vld int state=%d \n", fsm->b_sess_vld);
		}
		if (otg_int_src & OTGSC_INTSTS_B_SESSION_END) {
			fsm->b_sess_end =
			    (otg_sc & OTGSC_STS_B_SESSION_END) ? 1 : 0;
			trigger = 1;
			VDBG("!!!!b_sess_end int state=%d \n", fsm->b_sess_end);
		}
	}

	/* process USB interrupts */
	if ((usb_int_src & USB_STS_PORT_CHANGE)
	    && (fsm->protocol == OTG_FSM_PROTO_HOST)) {
		/* Device resume do not generate statemachine change */
		if (MCF_USB0_PORTSC & PORTSC_PORT_FORCE_RESUME) {
			if (otg->default_a) {
				fsm->b_bus_resume = 1;
				trigger = 1;
			} else {
				fsm->a_bus_resume = 1;
				trigger = 1;
			}
		}

		tmp = (MCF_USB0_PORTSC & PORTSC_CURRENT_CONNECT_STATUS) ? 1 : 0;
		if (otg->default_a && (fsm-> b_conn != tmp)) {
			fsm->b_conn = tmp;
			trigger = 1;
		} else if (!otg->default_a && (fsm->a_conn != tmp)) {
			fsm->a_conn = tmp;
			trigger = 1;
		}
	}
	/* Workaround: sometimes CSC bit will lost.  We change to
	 * polling CCS bit for connect change */
	if (fsm->protocol == OTG_FSM_PROTO_GADGET) {
		if (usb_int_src & USB_STS_DCSUSPEND) {
			VDBG("peripheral detected suspend \n");
			if (otg->default_a)
				/* A-device detects B suspend */
				fsm->b_bus_suspend = 1;
			else
				/* B-device detects A suspend */
				fsm->a_bus_suspend = 1;
			trigger = 1;
		} else if (usb_int_src & USB_STS_PORT_CHANGE) {
                      VDBG("peripheral resumed \n");
			if (otg->default_a)
				fsm->b_bus_suspend = 0;
			else
				fsm->a_bus_suspend = 0;
			trigger = 1;
		}
	}

	/* Invoke statemachine until state is stable */
	while (trigger) {
		trigger = otg_fsm_statemachine(fsm);
	}

	return IRQ_HANDLED;
}

static struct otg_fsm_ops fsl_otg_ops = {
	.chrg_vbus = fsl_otg_chrg_vbus,
	.drv_vbus = fsl_otg_drv_vbus,
	.loc_conn = fsl_otg_loc_conn,
	.loc_sof = fsl_otg_loc_sof,
	.start_pulse = fsl_otg_start_pulse,

	.add_timer = fsl_otg_add_timer,
	.del_timer = fsl_otg_del_timer,

	.start_host = fsl_otg_start_host,
	.start_gadget = fsl_otg_start_gadget,
};

/* Initialize the global variable fsl_otg_dev and request IRQ for OTG */
int fsl_otg_config(void)
{
	int status;
	struct fsl_otg *fsl_otg_tc;

	if (fsl_otg_dev)
		return 0;

	/* allocate space to fsl otg device */
	fsl_otg_tc = kmalloc(sizeof(struct fsl_otg), GFP_KERNEL);
	if (!fsl_otg_tc)
		return -ENODEV;
	memset(fsl_otg_tc, 0, sizeof(struct fsl_otg));

	INIT_LIST_HEAD(&active_timers);
	fsl_otg_init_timers(&fsl_otg_tc->fsm);

	/* Set OTG state machine operations */
	fsl_otg_tc->fsm.ops = &fsl_otg_ops;

	/* initialize the otg structure */
	fsl_otg_tc->otg.label = DRIVER_DESC;
	fsl_otg_tc->otg.set_host = fsl_otg_set_host;
	fsl_otg_tc->otg.set_peripheral = fsl_otg_set_peripheral;
	fsl_otg_tc->otg.set_power = fsl_otg_set_power;
	fsl_otg_tc->otg.start_hnp = fsl_otg_start_hnp;
	fsl_otg_tc->otg.start_srp = fsl_otg_start_srp;

	fsl_otg_dev = fsl_otg_tc;

	/* Store the otg transceiver */
	status = otg_set_transceiver(&fsl_otg_tc->otg);
	if (status)
		return status;

	return 0;
}

/* OTG Initialization*/
int usb_otg_start(struct platform_device *pdev)
{
	struct fsl_otg *p_otg;
	struct otg_transceiver *otg_trans = otg_get_transceiver();
	struct otg_fsm *fsm;
	int status;
	u32 temp;
	int timeout;

	p_otg = container_of(otg_trans, struct fsl_otg, otg);
	fsm = &p_otg->fsm;

	/* Initialize the state machine structure with default values */
	SET_OTG_STATE(otg_trans, OTG_STATE_UNDEFINED);
	fsm->transceiver = &p_otg->otg;

#define MCF532x_USB_OTG_INT_NUMBER		(128+47)
	p_otg->irq = MCF532x_USB_OTG_INT_NUMBER;
	status = request_irq(p_otg->irq, fsl_otg_isr,
				SA_SHIRQ|SA_INTERRUPT, DRIVER_DESC, p_otg);
	if (status) {
		dev_dbg(p_otg->otg.dev, "can't get IRQ %d, error %d\n",
			p_otg->irq, status);
		kfree(p_otg);
		return status;
	} else {
		/*
		* Set interrupt level:
		*/
		MCF_INTC1_ICR(p_otg->irq-128) = 0x1;
		/*
		* Clear Interrupt Mask corresponding bit:
		*/
		MCF_INTC1_CIMR = p_otg->irq-128;
	}


	/* stop the controller */
	MCF_USB0_USBCMD &= ~MCF_USB_USBCMD_RS;

	/* reset the controller */
	MCF_USB0_USBCMD = MCF_USB_USBCMD_RST;

	/* wait reset completed */
	timeout = 500;
	while (timeout-- && (MCF_USB0_USBCMD&0x2))
		udelay(1);
	if (timeout <=0)
		ERR("%s - USBCMD_RST never clear. Timeout is %d \n",
			__FUNCTION__, timeout);

	/* configurate the VBUSHS as IDLE(both host and device) */
	MCF_USB0_USBMODE = USB_MODE_STREAM_DISABLE;

	/* configurate PHY interface as ULPI */
	temp = MCF_USB0_PORTSC;
	temp &= ~(PORTSC_PHY_TYPE_SEL | PORTSC_PTW);
	temp |= PORTSC_PTS_ULPI;
	MCF_USB0_PORTSC = temp;


	/* disable all interrupt and clear all OTGSC status */
	temp = MCF_USB0_OTGSC;
	temp &= ~OTGSC_INTERRUPT_ENABLE_BITS_MASK;
	temp |= OTGSC_INTERRUPT_STATUS_BITS_MASK | OTGSC_CTRL_VBUS_DISCHARGE;
	MCF_USB0_OTGSC = temp;

	fsl_otg_drv_vbus(0);//DM FIXME


	return 0;
}

/* Initialize board specific registers,PIB board,clock and pin multiplexing */
static void board_init(void)
{
	/* Enable ULPI pins */
	MCF_GPIO_PAR_FEC = 0x0A;

	/* Enable the USBH_CLKIN pin */
	MCF_GPIO_PAR_IRQ = (MCF_GPIO_PAR_IRQ&~MCF_GPIO_PAR_IRQ_PAR_IRQ2(0x3))|
	MCF_GPIO_PAR_IRQ_PAR_IRQ2(0x2);

	/* Configure the drive strength for ULPI */
	MCF_GPIO_DSCR_FEC=0x00;

	/* USB_CLKIN pin drives USB serial interface clocks*/
	MCF_CCM_MISCCR &=~MCF_CCM_MISCCR_USBSRC;

	udelay(100);
	MCF_RCM_RCR|=MCF_RCM_RCR_FRCRSTOUT;
	udelay(100);
	MCF_RCM_RCR=0;
	udelay(100);

}

/*-------------------------------------------------------------------------
		PROC File System Support
-------------------------------------------------------------------------*/
#ifdef CONFIG_USB_OTG_DEBUG_FILES

#include <linux/seq_file.h>

static const char proc_filename[] = "driver/fsl_usb2_otg";

static int otg_proc_read(char *page, char **start, off_t off, int count,
			 int *eof, void *_dev)
{
	struct otg_fsm *fsm = &fsl_otg_dev->fsm;
	char *buf = page;
	char *next = buf;
	unsigned size = count;
	unsigned long flags;
	int t;
	u32 tmp_reg;

	if (off != 0)
		return 0;

	spin_lock_irqsave(&fsm->lock, flags);

	/* ------basic driver infomation ---- */
	t = scnprintf(next, size,
		      DRIVER_DESC "\n" "fsl_usb2_otg version: %s\n\n",
		      DRIVER_VERSION);
	size -= t;
	next += t;

	/* ------ Registers ----- */
	/* tmp_reg = le32_to_cpu(usb_dr_regs->otgsc); */
	tmp_reg = MCF_USB0_OTGSC;
	t = scnprintf(next, size, "OTGSC reg: %x\n", tmp_reg);
	size -= t;
	next += t;

	/* tmp_reg = le32_to_cpu(usb_dr_regs->portsc); */
	tmp_reg = MCF_USB0_PORTSC;
	t = scnprintf(next, size, "PORTSC reg: %x\n", tmp_reg);
	size -= t;
	next += t;

	/* tmp_reg = le32_to_cpu(usb_dr_regs->usbmode); */
	tmp_reg = MCF_USB0_USBMODE;
	t = scnprintf(next, size, "USBMODE reg: %x\n", tmp_reg);
	size -= t;
	next += t;

	/* tmp_reg = le32_to_cpu(usb_dr_regs->usbcmd); */
	tmp_reg = MCF_USB0_USBCMD;
	t = scnprintf(next, size, "USBCMD reg: %x\n", tmp_reg);
	size -= t;
	next += t;

	/* tmp_reg = le32_to_cpu(usb_dr_regs->usbsts); */
	tmp_reg = MCF_USB0_USBSTS;
	t = scnprintf(next, size, "USBSTS reg: %x\n", tmp_reg);
	size -= t;
	next += t;

	/* ------ State ----- */
	t = scnprintf(next, size,
		      "OTG state: %s\n\n",
		      otg_fsm_state_string(fsl_otg_dev->otg.state));
	size -= t;
	next += t;

	/* ------ State Machine Variables ----- */
	t = scnprintf(next, size, "a_bus_req: %d\n", fsm->a_bus_req);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_bus_req: %d\n", fsm->b_bus_req);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_bus_resume: %d\n", fsm->a_bus_resume);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_bus_suspend: %d\n", fsm->a_bus_suspend);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_conn: %d\n", fsm->a_conn);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_sess_vld: %d\n", fsm->a_sess_vld);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_srp_det: %d\n", fsm->a_srp_det);
	size -= t;
	next += t;

	t = scnprintf(next, size, "a_vbus_vld: %d\n", fsm->a_vbus_vld);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_bus_resume: %d\n", fsm->b_bus_resume);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_bus_suspend: %d\n", fsm->b_bus_suspend);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_conn: %d\n", fsm->b_conn);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_se0_srp: %d\n", fsm->b_se0_srp);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_sess_end: %d\n", fsm->b_sess_end);
	size -= t;
	next += t;

	t = scnprintf(next, size, "b_sess_vld: %d\n", fsm->b_sess_vld);
	size -= t;
	next += t;

	t = scnprintf(next, size, "id: %d\n", fsm->id);
	size -= t;
	next += t;

	spin_unlock_irqrestore(&fsm->lock, flags);

	*eof = 1;
	return count - size;
}

#define create_proc_file()	create_proc_read_entry(proc_filename, \
				0, NULL, otg_proc_read, NULL)

#define remove_proc_file()	remove_proc_entry(proc_filename, NULL)

#else				/* !CONFIG_USB_OTG_DEBUG_FILES */

#define create_proc_file()	do {} while (0)
#define remove_proc_file()	do {} while (0)

#endif				/*CONFIG_USB_OTG_DEBUG_FILES */

/*----------------------------------------------------------*/
/* Char driver interface to control some OTG input */

/* This function handle some ioctl command,such as get otg 
 * status and set host suspend
 */
static int fsl_otg_ioctl(struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{
	u32 retval = 0;

	switch (cmd) {
	case GET_OTG_STATUS:
		retval = fsl_otg_dev->host_working;
		break;

	case SET_A_SUSPEND_REQ:
		fsl_otg_dev->fsm.a_suspend_req = arg;
		break;

	case SET_A_BUS_DROP:
		fsl_otg_dev->fsm.a_bus_drop = arg;
		break;

	case SET_A_BUS_REQ:
		fsl_otg_dev->fsm.a_bus_req = arg;
		break;

	case SET_B_BUS_REQ:
		fsl_otg_dev->fsm.b_bus_req = arg;
		break;

	default:
		break;
	}

	otg_fsm_statemachine(&fsl_otg_dev->fsm);

	return retval;
}

static int fsl_otg_open(struct inode *inode, struct file *file)
{

	return 0;
}

static int fsl_otg_release(struct inode *inode, struct file *file)
{

	return 0;
}

static struct file_operations otg_fops = {
	.owner = THIS_MODULE,
	.llseek = NULL,
	.read = NULL,
	.write = NULL,
	.ioctl = fsl_otg_ioctl,
	.open = fsl_otg_open,
	.release = fsl_otg_release,
};

static int __init fsl_otg_probe(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	int status;

	if (!pdev)
		return -ENODEV;

	/* Initialize the clock,multiplexing pin and PHY interface(ULPI) */
	board_init();

	/* configurate the OTG */
	status = fsl_otg_config();

	if (status) {
		printk(KERN_INFO "Couldn't init OTG module\n");
		return -status;
	}

	/* start OTG */
	status = usb_otg_start(pdev);

	if (register_chrdev(FSL_OTG_MAJOR, FSL_OTG_NAME, &otg_fops)) {
		printk(KERN_WARNING FSL_OTG_NAME
		       ": unable to register FSL OTG device\n");
		return -EIO;
	}

	create_proc_file();
	return status;
}

static int __exit fsl_otg_remove(struct device *dev)
{

	otg_set_transceiver(NULL);
	free_irq(fsl_otg_dev->irq, fsl_otg_dev);

	kfree(fsl_otg_dev);

	remove_proc_file();

	unregister_chrdev(FSL_OTG_MAJOR, FSL_OTG_NAME);

	fsl_otg_dev = NULL;
	return 0;
}

struct device_driver fsl_otg_driver = {
	.name = (char *)otg_dr_name,
	.bus = &platform_bus_type,
	.probe = fsl_otg_probe,
	.remove = fsl_otg_remove,
};

/*-------------------------------------------------------------------------*/

static int __init fsl_usb_otg_init(void)
{
	printk(KERN_INFO "driver %s, %s\n", otg_dr_name, DRIVER_VERSION);
	return driver_register(&fsl_otg_driver);
}

static void __exit fsl_usb_otg_exit(void)
{
	driver_unregister(&fsl_otg_driver);
}

module_init(fsl_usb_otg_init);
module_exit(fsl_usb_otg_exit);

MODULE_DESCRIPTION(DRIVER_INFO);
MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_LICENSE("GPL");
